#' Estimating under-five mortality using Maternal age cohort-derived methods
#'
#' @description
#'
#' `r lifecycle::badge('stable')`
#'
#' `u5mr_cohort()` uses the maternal age cohort-derived methods (MAC) through summary
#' birth history information and maternal age (or time since first birth) to
#' calculate the under-five mortality estimates.
#'
#' @param data preprocessed data
#' @param women total number of women
#' @param child_born children ever born
#' @param child_dead children dead
#' @param agegrp age grouping or time since first birth
#' @param iso3 the `iso3` code of the country from which the survey data come
#' @param svy_year end of the survey
#'
#'
#' @details
#'
#' In this cohort-derived method, under-five mortality and reference time are estimated
#' through summary birth history information from the mothers and her age or time since
#' her first birth.
#'
#' In case sample weights are available for the mothers, final variables
#' should be multiplied by these weights before summarizing.
#'
#' \strong{Computational Procedure}
#'
#' Two formulas were used to quantify MAC method:
#'
#' \strong{\eqn{5q0} component}
#'
#' logit(\out{5q0<sub>ijk</sub>}) = \eqn{\beta}\out{<sub>0i</sub>} + \out{U<sub>ij</sub>} +
#' \eqn{\beta}\out{<sub>1i</sub>} x logit(\out{CD<sub>ijk</sub>} /
#' \out{CEB<sub>ijk</sub>}) + \eqn{\beta}\out{<sub>2i</sub>} x \out{CEB<sub>ijk</sub>} +
#' \eqn{\beta}\out{<sub>3i</sub>} x PR1 + \eqn{\beta}\out{<sub>4i</sub>} x PR2 +
#' \eqn{\epsilon}\out{<sub>ijk</sub>}
#'
#' where
#'
#' i = 5-year maternal age group \eqn{\epsilon} (15-19, 20-24, ... , 45-49)
#' j = country
#' k = survey
#' \out{CD<sub>i</sub>} = total dead children from maternal age group `i`
#' \out{CEB<sub>i</sub>} = total children ever born from maternal age group `i`
#' PR1 = ratio of parity among maternal age group 15-19 and parity among maternal age
#' group 20-24
#' PR2 = ratio of parity among maternal age group 20-24 and parity among maternal age
#' group 25-29
#' \out{U<sub>ij</sub>} = country-specific random effects
#'
#' All coefficients vary by maternal age group, as indicated by `i` and the random
#' effects also vary by country, as indicated by `j`.
#'
#'
#' \strong{Reference time component}
#'
#' \out{reftime<sub>ijk</sub>} = \eqn{\beta}\out{<sub>0i</sub>} +
#' \eqn{\beta}\out{<sub>1i</sub>} x (\out{CD<sub>ijk</sub>} /
#' \out{CEB<sub>ijk</sub>}) +
#' \eqn{\beta}\out{<sub>2i</sub>} x \out{CEB<sub>ijk</sub>} +
#' \eqn{\beta}\out{<sub>3i</sub>} x PR1 + \eqn{\beta}\out{<sub>4i</sub>} x PR2 +
#' \eqn{\epsilon}\out{<sub>ijk</sub>}
#'
#'
#'
#' @references
#'
#' Rajaratnam JK, Tran LN, Lopez AD, Murray CJL (2010) Measuring Under-Five Mortality: Validation of New Low-Cost Methods. PLOS Medicine 7(4): e1000253.
#' (\doi{10.1371/journal.pmed.1000253}{10.1371/journal.pmed.1000253})
#'
#' @return `data.frame`
#'
#' * `iso3` - total number of women
#' * `svy_year` - total number of children ever born
#' * `agegrp` - five-year age groups
#' * `ref_time`  - time between survey year and interpolated year
#' * `year`  - reference year
#' * `q5` - under-five mortality
#'
#' @examples
#'
#' ## Example using fake survey data from Cambodia
#' data(cambodia)
#' camb <- u5mr_cohort(cambodia, women = "women", child_born = "child_born",
#' child_dead = "child_dead", agegrp = "agegrp", iso3 = "KHM", svy_year = 1234)
#'
#' with(camb,
#'      plot(year, q5 * 1000, type = "b", pch = 19,
#'           col = "black", xlab = "Year", ylab = "U5MR per 1000 live births",
#'           main = paste0("Under-five mortality, q(5) in Bangladesh, estimated\n",
#'                        "using the maternal age cohort-derived method")))
#'
#' @export
u5mr_cohort <- function(data, women = "women",
                        child_born = "child_born", child_dead = "child_dead",
                        agegrp = "agegrp", iso3 = "KHM", svy_year = 2010) {
  agegrp <- data[[agegrp]]
  women <- data[[women]]
  child_born <- data[[child_born]]
  child_dead <- data[[child_dead]]

  pi <- child_born / women
  di <- child_dead / child_born
  logit_di <- log(di / (1 - di))

  pr1 <- pi[1] / pi[2]
  pr2 <- pi[2] / pi[3]
  re <- mac_re$re[mac_re$iso3 == iso3]
  ## if there is no country that matches, then re = 0
  if (length(re) == 0) re <- 0

  ## calculate logit 5q0 values
  logit_5q0 <- mac_5q0$coef_intercept + re + mac_5q0$coef_log_cdceb *
    logit_di + mac_5q0$coef_ceb * pi + mac_5q0$coef_pr1 * pr1 +
    mac_5q0$coef_pr2 * pr2
  ## convert back to normal values by exponentiating
  q5 <- exp(logit_5q0) / (1 + exp(logit_5q0))

  ## calculate reference time
  ref_time <- mac_reftime$coef_intercept + mac_reftime$coef_cdceb * di +
    mac_reftime$coef_ceb * pi + mac_reftime$coef_pr1 * pr1 +
    mac_reftime$coef_pr2 * pr2
  year <- svy_year - ref_time

  data.frame(iso3, svy_year, agegrp, ref_time, year, q5)
}


# HELPERS -----------------------------------------------------------------



mac_5q0 <- data.frame(
  agegrp = c('15-19', '20-24', '25-29', '30-34', '35-39', '40-44', '45-49'),
  coef_intercept = c(-0.6975273, 0.1258313, 0.0167552, -0.1034484, -0.0852121, -0.237591, 0.044768),
  coef_log_cdceb = c(0.5555944, 0.8853436, 0.9853121, 0.9679322, 0.9728206, 0.9951801, 1.01955),
  coef_ceb = c(2.573548, 0.2111931, 0.0523679, 0.0180773, 0.0147395, 0.0206229, 0.0016674),
  coef_pr1 = c(-3.923797, -0.0087424, 0.0705175, -0.0029778, 0.2246798, 0.162713, 0.8637883),
  coef_pr2 = c(-0.2960762, -1.110689, -0.2891437, -0.0033048, -0.2317203, -0.0566164, -0.8015343)
)

mac_reftime <- data.frame(
  agegrp = c('15-19', '20-24', '25-29', '30-34', '35-39', '40-44', '45-49'),
  coef_intercept = c(0.9432311, 2.533395, 4.303299, 7.268542, 11.42926, 15.48549, 19.90218),
  coef_cdceb = c(0.0015192, 0.0006606, -0.0017479, -0.0000676, -0.0012584, -0.002121, -0.001287),
  coef_ceb = c(0.1923932, 0.2812458, 0.1005495, -0.3103958, -0.4738352, -0.5497375, -0.5534297),
  coef_pr1 = c(0.7542738, 0.8857344, 0.2908919, 0.5750943, 1.502711, 3.027917, 6.079983),
  coef_pr2 = c(0.543375, 0.6180932, 2.843069, 4.691129, 4.441438, 3.965783, 0.9077534)
)

mac_re <- data.frame(
  agegrp = c("15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "15-19", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "20-24", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29", "25-29",
             "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "30-34", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39", "35-39",
             "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "40-44", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49", "45-49"),
  iso3 = c('ARM', 'AZE', 'BDI', 'BEN', 'BFA', 'BGD', 'BOL', 'BRA', 'BWA', 'CAF', 'CIV', 'CMR', 'COG', 'COL', 'COM', 'DOM', 'ECU', 'EGY', 'ERI', 'ETH', 'GAB', 'GHA', 'GIN', 'GTM', 'HND', 'HTI', 'IDN', 'IND', 'JOR', 'KAZ', 'KEN', 'KGZ', 'KHM', 'LBR', 'LKA', 'LSO', 'MAR', 'MDG', 'MEX', 'MLI', 'MOZ', 'MRT', 'MWI', 'NAM', 'NER', 'NGA', 'NIC', 'NPL', 'PAK', 'PER', 'PHL', 'PRY', 'RWA', 'SDN', 'SEN', 'SWZ', 'TCD', 'TGO', 'THA', 'TTO', 'TUN', 'TUR', 'TZA', 'UGA', 'UZB', 'VNM', 'YEM', 'ZAF', 'ZMB', 'ZWE', 'ARM', 'AZE', 'BDI', 'BEN', 'BFA', 'BGD', 'BOL', 'BRA', 'BWA', 'CAF', 'CIV', 'CMR', 'COG', 'COL', 'COM', 'DOM', 'ECU', 'EGY', 'ERI', 'ETH', 'GAB', 'GHA', 'GIN', 'GTM', 'HND', 'HTI', 'IDN', 'IND', 'JOR', 'KAZ', 'KEN', 'KGZ', 'KHM', 'LBR', 'LKA', 'LSO', 'MAR', 'MDG', 'MEX', 'MLI', 'MOZ', 'MRT', 'MWI', 'NAM', 'NER', 'NGA', 'NIC', 'NPL', 'PAK', 'PER', 'PHL', 'PRY', 'RWA', 'SDN', 'SEN', 'SWZ', 'TCD', 'TGO', 'THA', 'TTO', 'TUN', 'TUR', 'TZA', 'UGA', 'UZB', 'VNM', 'YEM', 'ZAF', 'ZMB', 'ZWE', 'ARM', 'AZE', 'BDI', 'BEN', 'BFA', 'BGD', 'BOL', 'BRA', 'BWA', 'CAF', 'CIV', 'CMR', 'COG', 'COL', 'COM', 'DOM', 'ECU', 'EGY', 'ERI', 'ETH', 'GAB', 'GHA', 'GIN', 'GTM', 'HND', 'HTI', 'IDN', 'IND', 'JOR', 'KAZ', 'KEN', 'KGZ', 'KHM', 'LBR', 'LKA', 'LSO', 'MAR', 'MDG', 'MEX', 'MLI', 'MOZ', 'MRT', 'MWI', 'NAM', 'NER', 'NGA', 'NIC', 'NPL', 'PAK', 'PER', 'PHL', 'PRY', 'RWA', 'SDN', 'SEN', 'SWZ', 'TCD', 'TGO', 'THA', 'TTO', 'TUN', 'TUR', 'TZA', 'UGA', 'UZB', 'VNM', 'YEM', 'ZAF', 'ZMB', 'ZWE', 'ARM', 'AZE', 'BDI', 'BEN', 'BFA', 'BGD',
           'BOL', 'BRA', 'BWA', 'CAF', 'CIV', 'CMR', 'COG', 'COL', 'COM', 'DOM', 'ECU', 'EGY', 'ERI', 'ETH', 'GAB', 'GHA', 'GIN', 'GTM', 'HND', 'HTI', 'IDN', 'IND', 'JOR', 'KAZ', 'KEN', 'KGZ', 'KHM', 'LBR', 'LKA', 'LSO', 'MAR', 'MDG', 'MEX', 'MLI', 'MOZ', 'MRT', 'MWI', 'NAM', 'NER', 'NGA', 'NIC', 'NPL', 'PAK', 'PER', 'PHL', 'PRY', 'RWA', 'SDN', 'SEN', 'SWZ', 'TCD', 'TGO', 'THA', 'TTO', 'TUN', 'TUR', 'TZA', 'UGA', 'UZB', 'VNM', 'YEM', 'ZAF', 'ZMB', 'ZWE', 'ARM', 'AZE', 'BDI', 'BEN', 'BFA', 'BGD', 'BOL', 'BRA', 'BWA', 'CAF', 'CIV', 'CMR', 'COG', 'COL', 'COM', 'DOM', 'ECU', 'EGY', 'ERI', 'ETH', 'GAB', 'GHA', 'GIN', 'GTM', 'HND', 'HTI', 'IDN', 'IND', 'JOR', 'KAZ', 'KEN', 'KGZ', 'KHM', 'LBR', 'LKA', 'LSO', 'MAR', 'MDG', 'MEX', 'MLI', 'MOZ', 'MRT', 'MWI', 'NAM', 'NER', 'NGA', 'NIC', 'NPL', 'PAK', 'PER', 'PHL', 'PRY', 'RWA', 'SDN', 'SEN', 'SWZ', 'TCD', 'TGO', 'THA', 'TTO', 'TUN', 'TUR', 'TZA', 'UGA', 'UZB', 'VNM', 'YEM', 'ZAF', 'ZMB', 'ZWE', 'ARM', 'AZE', 'BDI', 'BEN', 'BFA', 'BGD', 'BOL', 'BRA', 'BWA', 'CAF', 'CIV', 'CMR', 'COG', 'COL', 'COM', 'DOM', 'ECU', 'EGY', 'ERI', 'ETH', 'GAB', 'GHA', 'GIN', 'GTM', 'HND', 'HTI', 'IDN',
           'IND', 'JOR', 'KAZ', 'KEN', 'KGZ', 'KHM', 'LBR', 'LKA', 'LSO', 'MAR', 'MDG', 'MEX', 'MLI', 'MOZ', 'MRT', 'MWI', 'NAM', 'NER', 'NGA', 'NIC', 'NPL', 'PAK', 'PER', 'PHL', 'PRY', 'RWA', 'SDN', 'SEN', 'SWZ', 'TCD', 'TGO', 'THA', 'TTO', 'TUN', 'TUR', 'TZA', 'UGA', 'UZB', 'VNM', 'YEM', 'ZAF', 'ZMB', 'ZWE', 'ARM', 'AZE', 'BDI', 'BEN', 'BFA', 'BGD', 'BOL', 'BRA', 'BWA', 'CAF', 'CIV', 'CMR', 'COG', 'COL', 'COM', 'DOM', 'ECU', 'EGY', 'ERI', 'ETH', 'GAB', 'GHA', 'GIN', 'GTM', 'HND', 'HTI', 'IDN', 'IND', 'JOR', 'KAZ', 'KEN', 'KGZ', 'KHM', 'LBR', 'LKA', 'LSO', 'MAR', 'MDG', 'MEX', 'MLI', 'MOZ', 'MRT', 'MWI', 'NAM', 'NER', 'NGA', 'NIC', 'NPL', 'PAK', 'PER', 'PHL', 'PRY', 'RWA', 'SDN', 'SEN', 'SWZ', 'TCD', 'TGO', 'THA', 'TTO', 'TUN', 'TUR', 'TZA', 'UGA', 'UZB', 'VNM', 'YEM', 'ZAF', 'ZMB', 'ZWE'),
  re = c(-0.0740473806025451, -0.795791117654689, 0.0155759526158037, 0.167607916035274,
         0.312169911719107, -0.296688023813686, -0.0347382273558816, -0.0807181991540473,
         -0.00587074151497604, 0.0341199997542009, 0.0987730517784896, 0.214199601121983,
         0.101434384497789, -0.252926194970638, 0.201941057537439, -0.335415771831015,
         -0.13894049257474, -0.17633911342114, 0.308017811901251, 0.0622887918523105,
         0.0495851202778916, 0.348063608603817, 0.269097292481718, -0.0822907533315262,
         -0.45401938813222, 0.242555390947918, -0.0414108708084773, -0.235634428567539,
         -0.410657587622013, 0.461005429816519, -0.114078617228897, 0.124042825789835,
         0.112012316556841, 0.177191559784913, -0.533245228365606, 0.244690752241358,
         -0.253770138470807, 0.0427130903724443, -0.151949500816286, 0.221563125229524,
         0.121359808817806, 0.251304137747192, 0.141382927962772, 0.0979680619614078,
         0.362533204668132, 0.468893124782236, -0.238506315601834, -0.215977145004483,
         -0.268123473779954, -0.131684862494324, -0.256291791064001, -0.503895050382619,
         0.435827395207834, -0.00818867178546823, 0.202913086086713, 0.388892640993358,
         0.116454974260369, 0.239297053094394, 0.206897543532596, -0.444633894827663,
         0.102355012116847, -0.265088057216033, 0.171283416422014, 0.068321016313176,
         -0.399809598084503, -0.0282595054955628, -0.113126794403926, 0.0296563717984058,
         0.278059822173844, -0.149931652478513, 0.0226273023933315, 0.191677673151666,
         0.0793470615451354, 0.0432160268164751, 0.0454145682561387, -0.00101334949487753,
         -0.125846269439958, -0.105600603498696, -0.0327436405540945, 0.0261922848616024,
         -0.0519554128785404, -0.0107890447217732, 0.0735177332022234, -0.0613379469231933,
         -0.049151442842105, -0.183711910937571, 0.0114849866708276, -0.0301466123730515,
         0.133021245486241, -0.0241454496412133, -0.0193380761575792, 0.0156187626309521,
         0.115575865490356, -0.0993359505210646, -0.0674513480183834, -0.034396588453635,
         0.065713848824011, -0.0172355741140534, -0.234405361204415, 0.116528251911681,
         -0.0268324932077755, -0.0793378313527664, 0.126938872898581, 0.0190697557242236,
         0.128202523894199, 0.0743618838667504, -0.0959220218156416, 0.016882362050265,
         0.0343723054707976, 0.0841920122399618, 0.02395193795347, -0.0461280053433558,
         0.0220294979939325, -0.0477751271316608, 0.0804192327632425, 0.0724335794469876,
         -0.10165309023505, -0.0274375729086623, -0.0611557607310711, -0.0304699373537955,
         0.0298093822994429, -0.0326374950384448, 0.11576243813471, 0.0443548037177634,
         0.0341292762656796, 0.0368911447919037, 0.0553273076332881, 0.0917918290311791,
         0.0158163825765856, -0.124905822245882, 0.0802488977826972, -0.0363509489142704,
         0.0149236193472427, -0.00393416620249638, -0.0202978944239627, -0.0184352466286266,
         -0.0651257288710919, -0.0610401180676931,
         -0.0206686371700154, -0.09313217770706, 0.000000000000000974362352833778,
         -0.0000000000000000185775251266684, 0.00000000000000138479360740729,
         0.000000000000000549072627552913, 0.000000000000000413951629330078,
         -0.00000000000000189109624246481, -0.00000000000000300753768234173,
         0.0000000000000000643404820606904, -0.00000000000000182363694136833,
         -0.000000000000000257296646105201, -0.000000000000000847129073767889,
         -0.00000000000000118499385614645, 0.0000000000000000153996482528334,
         -0.000000000000000179357051733884, 0.000000000000000126132164673102,
         -0.000000000000000950559442671591, -0.000000000000000163200682410404,
         0.00000000000000117082306186797, 0.000000000000000369004439345049,
         0.000000000000000268552488382658, -0.0000000000000000586365934614987,
         0.00000000000000101614523952222, 0.00000000000000150870517914443,
         -0.0000000000000010254380443248, 0.000000000000000563928981943078,
         -0.000000000000000566764495128108, -0.000000000000000783039678917568,
         -0.0000000000000000836744295903942, -0.00000000000000343394482052709,
         -0.00000000000000235192116951888, -0.000000000000000554167099235983,
         -0.000000000000000614196493376823, 0.00000000000000143731973434961,
         -0.000000000000000427699708230177, 0.00000000000000131091438726934,
         -0.00000000000000111284584112205, -0.00000000000000117034332331515,
         0.000000000000000281859812471901, 0.000000000000000692023588332105,
         0.00000000000000259281230752607, -0.000000000000000816597805707923,
         -0.00000000000000103141009885278, 0.000000000000000102968721249683,
         0.00000000000000272161792133958, 0.00000000000000132589993134094,
         0.0000000000000018996442806542, -0.00000000000000137444397262512,
         0.00000000000000122399094387007, -0.00000000000000273021954752028,
         0.00000000000000178070560035333, 0.00000000000000209054292177263,
         -0.0000000000000000172320021878189, 0.00000000000000221551767517667,
         0.000000000000000203227444143104, 0.000000000000000419996453782939,
         -0.000000000000000972928804211034, -0.000000000000000570479057214518,
         0.00000000000000061654725350333, 0.00000000000000198714974309453,
         0.000000000000000289834787952045, 0.00000000000000118221933999523,
         0.00000000000000138723464398358, 0.000000000000000940943820552307,
         0.000000000000000603980410562396, -0.00000000000000198970994417312,
         0.000000000000000766282794075871, -0.000000000000000593223968713127,
         -0.000000000000000268783431767057, 0.000000000000000680345446852395,
         -0.00000000000000432382153146059, -0.00300181529677665, -0.00129311188812351,
         0.0173948026213881, 0.000741303614537233, -0.000832822469943408, -0.0149893019964271,
         -0.0204566697267605, 0.001560270537412, 0.000796823471862619, -0.000205802334074667,
         0.00489285064595247, -0.00936365236299378, 0.00863680389594218, -0.00313084395883648,
         -0.00586247989100341, -0.0167150145385607, -0.00756090271378879, 0.0220449914399241,
         0.00261387902460977, -0.0039105200315521, -0.00532391329605428, -0.00332361156245709,
         0.00408683083958631, 0.0052910573930709, -0.00160877875844504, 0.00533302196083457,
         -0.00212915362605054, -0.0119273254678541, -0.00755110048055948, 0.000848096895951275,
         -0.0202850395062935, -0.00136957991981613, 0.0101868539464486, 0.00739588693906935,
         0.00723006322064839, -0.0087617637339176, 0.00196022863165025, -0.0026874824685028,
         -0.00835638128063606, 0.0098556937093614, 0.000800224455882823, 0.00189502418283973,
         0.00890281235695304, -0.00147704077009572, 0.00881319262254804, 0.00639088833290026,
         -0.00407596033273286, -0.00398738680035375, -0.0127720697954296, 0.0100280404744035,
         -0.00298647705787444, 0.0107168264190199, 0.0046457977174518, -0.00108429243551952,
         0.00853493863842387, 0.00281985190241627, 0.00216681200617402, -0.00343699004568628,
         -0.0012758643592932, 0.0135953231264717, 0.00573532789231331, -0.010945572538109,
         0.00942733056598776, 0.00372480044890102, 0.00153753468725453, 0.00946555282915329,
         -0.00601493641287829, -0.00292102915695526, -0.000410201498194045, -0.00803484893479638,
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -0.0432453798170324, 0.00560134714370365,
         0.0249521248433512, -0.0155738563600888, -0.0212386224152112,
         -0.022235394395627, -0.0357290093324093, 0.0664135151803871, 0.0145265765526581,
         -0.0114131467893674, -0.0405658177586163, -0.066868387189781, -0.062836355793198,
         0.0317497072507178, -0.030984508227835, 0.0328780031509164, 0.100216368073902,
         0.0880410784873099, -0.0191080542081191, -0.0370484739499445, -0.105152424649484,
         0.0401787723807475, 0.00320861174247859, 0.070110668131293, -0.0396251195503589,
         0.0788853579645186, -0.00317150592723563, -0.027479012195949, -0.0387677308243425,
         0.118328799117877, -0.0425085405483079, 0.0800965184165024, -0.0695944368957809,
         0.0676141006163097, -0.0511004878653918, -0.0193579165831984, 0.0482029461302877,
         0.0247676766012574, 0.0334838852580998, 0.0296267797854918, 0.0630856750607265,
         -0.0510485158841582, 0.0409393697637698, 0.0391124777751977, 0.0649240805058892,
         -0.0181805180860715, -0.119043352951846, 0.0162244799459579, 0.0531087653168386,
         0.0372205349685101, 0.0286088658079497, 0.0495226431349513, -0.0817897920820907,
         -0.0550754578194008, -0.0173507323465471, -0.233359007285683, -0.0200965193440001,
         -0.0257971261778375, -0.0255853450503265, 0.00647100123906467, 0.0318723146782486,
         0.0160438492107449, -0.0208932257641941, -0.0301389877507251, 0.0484910844562105,
         0.0390538656561936, 0.073374159746082, -0.0275506681624441, -0.0152452314243491,
         -0.0221773426872071, -0.00000000658142803654897, 0.00000000549001592993776,
         0.0000000198668591825604, 0.00000000197995328154996, 0.0000000448093576761179,
         0.000000000169939626248278, 0.0000000192195042644619, 0.0000000630062388694952,
         0.000000117132438271513, -0.0000000598742144712409, -0.0000000806352906505957,
         -0.0000000780524006437864, -0.000000120670454488179, -0.0000000438402718308179,
         0.0000000566560029230225, 0.000000110416386162676, 0.0000000428397241626569,
         0.000000179741440726228, -0.0000000093390309668003, -0.0000000229723188360878,
         -0.0000000667412279715967, 0.000000107303954089799, 0.00000000726226116061813,
         0.0000000204651542059649, -0.00000000931714564373454, 0.0000000260977268342249,
         -0.0000000464800393356342, -0.0000000406485729182893, -0.0000000975574631004335,
         0.000000037495526370434, -0.000000216334914430975, 0.0000000672466193275683,
         -0.0000000704750921453968, 0.0000000557289692336193, -0.0000000634406485811656,
         -0.000000111121851964046, 0.0000000714285958987262, 0.0000000903404887561128,
         0.00000000601874472391811, 0.000000113834365478908, -0.0000000252773183735804,
         0.0000000422100272933371, 0.0000000530364394390656, 0.0000000694030549637975,
         0.000000128680694103497, -0.0000000335656290890042, -0.000000082676743810319,
         -0.00000000780778548620502, 0.0000000814160259104205, 0.000000130238700617509,
         0.0000000371865678372784, 0.000000000283632031076552, -0.000000220653175685402,
         0.0000000147793572008302, 0.0000000940275214266338, -0.000000251933753016487,
         -0.0000000626644220445201, 0.00000000834203637173992, -0.000000053818817085844,
         0.0000000295402880926745, 0.0000000226628658428821, -0.0000000429021714301435,
         -0.00000000988591405273897, 0.00000000214314254121632, 0.0000000993402384095545,
         -0.00000000735372412943671, 0.0000000784846845721814, 0.0000000331550688313437,
         -0.000000172322181425369, -0.0000000745366109870467)
)

