################################################################################
# ---------------------------- SDF.TEST FUNCTIONS ---------------------------- #
################################################################################

#' Null distribution estimator
#'
#' Estimates the null distribution of S given X1.
#'
#' @param X1      Longest time series.
#' @param n2      Integer. Length of the shortest time series \code{X2}.
#' @param Te      Integer. Number of bins.
#' @param m1      Numeric. Approximate number of X1 observations per bin.
#' @param m2      Numeric. Approximate number of X2 observations per bin.
#' @param q1      Integer. Penalization order for \code{X1}.
#' @param method1 Character. Method for selecting the smoothing parameter for
#'                \code{X1}.
#' @param f1_true Vector or \code{NULL}. True regression function evaluated at
#'                equi-spaced points. Required only if \code{method1} is set to
#'                "\code{GCV-oracle}" or "\code{ML-oracle}".
#' @param q2      Integer. Penalization order for \code{X2}.
#' @param method2 Character. Method for selecting the smoothing parameter for
#'                \code{X2}.
#' @param f2_true Vector or \code{NULL}. True regression function evaluated at
#'                equi-spaced points. Required only if \code{method2} is set to
#'                "\code{GCV-oracle}" or "\code{ML-oracle}".
#' @param D1      Matrix. DCT-I matrix of dimension \code{n1}.
#' @param D2      Matrix. DCT-I matrix of dimension \code{n2}.
#' @param evals1  eigenvalues of the \code{q1}-th order roughness penalty opera-
#'                tor in the Demmler-Reinsch basis.
#' @param evals2  eigenvalues of the \code{q2}-th order roughness penalty opera-
#'                tor in the Demmler-Reinsch basis.
#' @param N       Integer. Number of iterations for empirical null distribution
#'                computation.
#' @param cores   Integer. Number of cores for parallel computation of null dis-
#'                tribution.
#'
#' @return Vector of empirical null distribution of the test \code{S} given
#'         \code{X1}.
#'
#' @keywords internal
#' @noRd
nulldistr_estimator <- function(
    X1, n2, Te, m1, m2,
    q1, method1, f1_true,
    q2, method2, f2_true,
    D1, D2,
    evals1, evals2,
    N, cores
) {

  ### --- Input checks ---
  check_integer_scalar(N)
  check_integer_scalar(cores)

  ### --- variables setting ---
  n1 <- length(X1)
  shift1 <- (digamma(m1/2) - log(m1/2))/sqrt(2)
  shift2 <- (digamma(m2/2) - log(m2/2))/sqrt(2)

  ### --- Underlying spectral density ---
  logsdf <- logsdf_estimator(
    X = X1, Te = Te, m = m1,
    shift = 0, q = q1, D = D1,
    method = method1, f_true = f1_true, evals = evals1
  )
  sdf <- (m1/2)*exp(sqrt(2)*logsdf-digamma(m1/2))
  l <- length(sdf)

  x1_temp <- round(seq(1, round(l/2), length = n1))
  sdf1_temp <- sdf[x1_temp]
  acf1 <- sdf2acf(sdf1_temp)
  Sigma1 <- toeplitz(acf1)
  R1 <- t(chol(Sigma1))

  x2_temp <- round(seq(1, round(l/2), length = n2))
  sdf2_temp <- sdf[x2_temp]
  acf2 <- sdf2acf(sdf2_temp)
  Sigma2 <- toeplitz(acf2)
  R2 <- t(chol(Sigma2))

  ### --- Monte Carlo sampling ---
  if (cores==1) {
    ### --- No parallel ---
    S <- rep(0,N)
    Z1 <- matrix(rnorm(n1 * N), n1, N)
    Z2 <- matrix(rnorm(n2 * N), n2, N)
    for (i in 1:N) {
      X1_temp <- R1%*%Z1[, i]
      X2_temp <- R2%*%Z2[, i]
      g1_temp <- logsdf_estimator(
        X = X1_temp, Te = Te, m = m1,
        shift = shift1, q = q1, D = D1,
        method = method1, f_true = f1_true, evals = evals1
      )
      g2_temp <- logsdf_estimator(
        X = X2_temp, Te = Te, m = m2,
        shift = shift2, q = q2, D = D2,
        method = method2, f_true = f2_true, evals = evals2
      )
      S[i] <- mean((g1_temp - g2_temp)^2)
    }
  } else {
    ### --- Parallel backend ---
    `%dopar%` <- foreach::`%dopar%`
    cl <- parallel::makeCluster(cores)
    on.exit(parallel::stopCluster(cl), add = TRUE)
    doParallel::registerDoParallel(cl)

    ### --- Iterations ---
    S <- foreach::foreach(n=1:N, .combine = c) %dopar% {
      X1_temp <- R1%*%rnorm(n1)
      X2_temp <- R2%*%rnorm(n2)
      g1_temp <- logsdf_estimator(
        X = X1_temp, Te = Te, m = m1,
        shift = shift1, q = q1, D = D1,
        method = method1, f_true = f1_true, evals = evals1
      )
      g2_temp <- logsdf_estimator(
        X = X2_temp, Te = Te, m = m2,
        shift = shift2, q = q2, D = D2,
        method = method2, f_true = f2_true, evals = evals2
      )
      return(mean((g1_temp - g2_temp)^2))
    }
  }

  ### --- Output ---
  return(S)
}

#' SDF Test
#'
#' Performs the test for equality of spectral densities of two time series.
#'
#' @param X1        Longest time series.
#' @param X2        Shortest time series.
#' @param Te        Integer. Number of bins.
#' @param alpha     Numeric. Significance level for the test.
#' @param q1        Integer. Penalisation order for \code{X1},
#'                  \code{q=1,2,3,4,5,6} are available (default \code{4}).
#' @param method1   Character. Method for selecting the smoothing parameter for
#'                  \code{X1} (default "\code{GCV}").
#' @param f1_true   Vector or \code{NULL}. True regression function evaluated at
#'                  equi-spaced points. Required only if \code{method1} is set
#'                  to "\code{GCV-oracle}" or "\code{ML-oracle}".
#' @param q2        Integer. Penalization order for \code{X2} (default
#'                  \code{q1}).
#' @param method2   Character. Method for selecting the smoothing parameter for
#'                  \code{X2} (default \code{method1}).
#' @param f2_true   Vector or \code{NULL}. True regression function evaluated at
#'                  equi-spaced points. Required only if \code{method2} is set
#'                  to "\code{GCV-oracle}" or "\code{ML-oracle}" (default
#'                  \code{f1_true}).
#' @param N         Integer. Number of iterations for null distribution computa-
#'                  tion (default \code{10000}).
#' @param cores     Integer. Number of cores for parallel computation of null
#'                  distribution (default \code{1}).
#' @param nulldistr Vector or \code{NULL}. Vector of empirical null distribution
#'                  (default \code{NULL}).
#'
#' @return An object of class \code{"test"}, which is a list containing the
#'         following components:
#'         \itemize{
#'         \item \code{result}: Logical. \code{TRUE} if the null hypothesis of
#'              equality of spectral densities is accepted by the test,
#'              \code{FALSE} otherwise.
#'         \item \code{S}: Value of the statistic.
#'         \item \code{quantile}: Empirical quantile used for the test.
#'         \item \code{p-value}: p-value of the test according to the
#'              empirical null distribution.
#'         \item \code{nulldistr}: Vector of empirical null distribution.
#'         }
#'
#' @examples
#' # --- Example 1: Using random data ---
#' test <- sdf.test(
#'   X1 = rnorm(100), X2 = rnorm(80),
#'   q1 = 4, method1 = "GCV", Te = 20, alpha = 0.05, N = 1000
#' )
#' test
#'
#' # --- Example 2: Using random data ---
#' \donttest{
#' X1 <- arima.sim(list(order=c(1,0,0),ar=0.5),n = 1200,rand.gen = rnorm, sd = 1)
#' X2 <- arima.sim(list(order=c(1,0,0),ar=0.8),n = 1000,rand.gen = rnorm, sd = 1)
#' sdf.test(
#'   X1 = X1, X2 = X2,
#'   q1 = 4, method1 = "GCV", Te = 176, alpha = 0.05
#' )
#' }
#' 
#' # --- Example 3: Using EEG dataset ---
#' \donttest{
#' data(eeg_data)
#' X1 <- subset(eeg_data, condition == "first_frontal_tDCS")$signal
#' X2 <- subset(eeg_data, condition == "second_frontal_tDCS")$signal
#' test <- sdf.test(
#'   X1 = X1, X2 = X2,
#'   q1 = 4, method1 = "ML", Te = 2100, alpha = 0.05, cores = 50
#' )
#' test
#' }
#'
#' @references
#' Nadin, Krivobokova, Enikeeva (2026). Nonparametric two sample test of
#' spectral densities.
#' \url{https://arxiv.org/abs/2602.10774}
#'
#' @export
sdf.test <- function(
    X1, X2, Te, alpha,
    q1 = 4, method1 = "GCV", f1_true = NULL,
    q2 = q1, method2 = method1, f2_true = f1_true,
    N = 10000, cores = 1,
    nulldistr = NULL
){

  ### --- Input checks ---
  X1 <- check_numeric_vector(X1, "X1")
  X2 <- check_numeric_vector(X2, "X2")
  check_significance_level(alpha)
  check_penalisation_order(q1)
  check_penalisation_order(q2)
  check_integer_scalar(Te)
  check_method(x = method1, namex = "method1", f = f1_true, namef = "f1_true")
  check_method(x = method2, namex = "method2", f = f2_true, namef = "f2_true")

  n1 <- length(X1)
  n2 <- length(X2)
  if (length(X1) < length(X2)) {
    stop("'X1' must be the longest time series.")
  }
  if (log(n1)/log(n2) > 3/2) {
    stop("Length difference between 'X1' and 'X2' is too large.")
  }

  if (floor(n2/Te) < 2) {
    stop("'Te' too large.")
  }

  if (Te <= 1) {
    stop("'Te' too small.")
  }

  ### --- Variable setting ---
  m1 <- round(n1/Te)
  m2 <- round(n2/Te)
  D1 <- DCT.matrix(n1)
  D2 <- DCT.matrix(n2)
  TT <- 2*Te-2
  x <- seq(1,TT)/TT
  shift1 <- (digamma(m1/2) - log(m1/2))/sqrt(2)
  shift2 <- (digamma(m2/2) - log(m2/2))/sqrt(2)
  evals1 <- (2*pi*1:TT)^(2*q1)*sinc(pi*x)^(2*q1)/Q_pminus1(2*q1-2, x)
  evals2 <- (2*pi*1:TT)^(2*q2)*sinc(pi*x)^(2*q2)/Q_pminus1(2*q2-2, x)

  ### --- Empirical null distribution ---
  if (is.null(nulldistr)) {
    nulldistr <- nulldistr_estimator(
      X1 = X1, n2 = n2, Te = Te, m1 = m1, m2 = m2,
      q1 = q1, method1 = method1, f1_true = f1_true,
      q2 = q2, method2 = method2, f2_true = f2_true,
      D1 = D1, D2 = D2,
      evals1 = evals1, evals2 = evals2,
      N = N, cores = cores
    )
  } else {
    check_numeric_vector(nulldistr)
  }

  ### --- Test statistic ---
  g1 <- logsdf_estimator(
    X = X1, Te = Te, m = m1,
    shift = shift1, q = q1, D = D1,
    method = method1, f_true = f1_true, evals = evals1
  )
  g2 <- logsdf_estimator(
    X = X2, Te = Te, m = m2,
    shift = shift2, q = q2, D = D2,
    method = method2, f_true = f2_true, evals = evals2
  )
  S <- mean((g1 - g2)^2)

  ### --- Output ---
  q_crit <- quantile(nulldistr, 1 - alpha)
  pval <- 1 - ecdf(nulldistr)(S)
  result <- S<q_crit

  test <- list(
    result = result,
    S = S,
    quantile = q_crit,
    pval = pval,
    nulldistr = nulldistr)
  class(test) <- "test"
  return(test)
}

#' Print test
#'
#' Custom printing function for the class "test".
#'
#' @param x   Object of class "test".
#' @param ... Further arguments passed to or from other methods.
#'
#' @return Invisible \code{x}.
#'
#' @examples
#' test <- sdf.test(
#'   X1 = rnorm(50), X2 = rnorm(50),
#'   q1 = 2, method1 = "GCV", Te = 10, alpha = 0.05, N = 1000
#' )
#' print(test)
#' test
#'
#' @export
print.test <- function(x, ...) {
  cat("\n\t Test for Equality of Spectral Densities \n\n")
  cat("statistic = ", format(x$S, digits = 4),
      ", quantile = ", format(x$quantile, digits = 4),
      ", p-value = ", format.pval(x$pval, digits = 4), "\n", sep = "")
  cat("null hypothesis: equality of spectral densities\n\n")
  if (x$result) {
    cat("Result: fail to reject the null hypothesis\n")
  } else {
    cat("Result: reject the null hypothesis\n")
  }
  invisible(x)
}
