% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/elo_model.R
\name{fit_elo_model}
\alias{fit_elo_model}
\title{Fit an EloChoice model to pairwise comparison data}
\usage{
fit_elo_model(elo_data, runs = 5, verbose = FALSE, ...)
}
\arguments{
\item{elo_data}{A data frame or tibble containing \code{winner} and
\code{loser} columns. Typically produced using
\code{\link{build_elo_data}}.}

\item{runs}{Integer number of randomizations to use in
\code{EloChoice::elochoice}. Default is 5.}

\item{verbose}{Logical. If \code{TRUE} (default), show any messages/warnings
emitted by the underlying fitting functions. If \code{FALSE}, suppress
noisy output to keep examples and reports clean.}

\item{...}{Additional arguments passed to
\code{EloChoice::elochoice()}.}
}
\value{
A named list with components:
\describe{
\item{engine}{Character scalar identifying the scoring engine
(\code{"EloChoice"}).}
\item{fit}{The \code{"elochoice"} model object.}
\item{elo}{A tibble with columns \code{ID} and \code{elo}.}
\item{reliability}{Numeric scalar: mean unweighted reliability index.}
\item{reliability_weighted}{Numeric scalar: mean weighted
reliability index.}
}
}
\description{
This function fits an Elo-based paired-comparison model using the
\pkg{EloChoice} package. It is intended to complement
\code{\link{fit_bt_model}} by providing an alternative scoring framework
based on Elo ratings rather than Bradley–Terry models.
}
\details{
The input \code{elo_data} must contain two columns:
\enumerate{
\item \code{winner}: ID of the winning sample in each pairwise trial
\item \code{loser}: ID of the losing sample in each trial
}
These can be created from standard pairwise comparison output using
\code{\link{build_elo_data}}.

Internally, this function calls:
\itemize{
\item \code{\link[EloChoice]{elochoice}} — to estimate Elo ratings using
repeated randomization of trial order;
\item \code{\link[EloChoice]{reliability}} — to compute unweighted and
weighted reliability indices as described in Clark et al. (2018).
}

If the \pkg{EloChoice} package is not installed, a helpful error message
is shown telling the user how to install it.

The returned object mirrors the structure of \code{\link{fit_bt_model}}
for consistency across scoring engines:
\itemize{
\item \code{engine} — always \code{"EloChoice"}.
\item \code{fit} — the raw \code{"elochoice"} object returned by
\code{EloChoice::elochoice()}.
\item \code{elo} — a tibble with columns:
\itemize{
\item \code{ID}: sample identifier
\item \code{elo}: estimated Elo rating
}
(Unlike Bradley–Terry models, EloChoice does not provide
standard errors for these ratings, so none are returned.)
\item \code{reliability} — the mean unweighted reliability index
(mean proportion of “upsets” across randomizations).
\item \code{reliability_weighted} — the mean weighted reliability index
(weighted version of the upset measure).
}
}
\examples{
data("example_writing_pairs", package = "pairwiseLLM")

elo_data <- build_elo_data(example_writing_pairs)

fit <- fit_elo_model(elo_data, runs = 5, verbose = FALSE)
fit$elo
fit$reliability
fit$reliability_weighted

}
\references{
Clark AP, Howard KL, Woods AT, Penton-Voak IS, Neumann C (2018).
"Why rate when you could compare? Using the 'EloChoice' package to assess
pairwise comparisons of perceived physical strength."
\emph{PLOS ONE}, 13(1), e0190393.
\doi{10.1371/journal.pone.0190393}.
}
