% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/anthropic_live.R
\name{anthropic_compare_pair_live}
\alias{anthropic_compare_pair_live}
\title{Live Anthropic (Claude) comparison for a single pair of samples}
\usage{
anthropic_compare_pair_live(
  ID1,
  text1,
  ID2,
  text2,
  model,
  trait_name,
  trait_description,
  prompt_template = set_prompt_template(),
  tag_prefix = "<BETTER_SAMPLE>",
  tag_suffix = "</BETTER_SAMPLE>",
  api_key = NULL,
  anthropic_version = "2023-06-01",
  reasoning = c("none", "enabled"),
  include_raw = FALSE,
  include_thoughts = NULL,
  ...
)
}
\arguments{
\item{ID1}{Character ID for the first sample.}

\item{text1}{Character string containing the first sample's text.}

\item{ID2}{Character ID for the second sample.}

\item{text2}{Character string containing the second sample's text.}

\item{model}{Anthropic Claude model name (for example
\code{"claude-sonnet-4-5"}, \code{"claude-haiku-4-5"},
or \code{"claude-opus-4-5"}).}

\item{trait_name}{Short label for the trait (for example "Overall Quality").}

\item{trait_description}{Full-text definition of the trait.}

\item{prompt_template}{Prompt template string, typically from
\code{\link{set_prompt_template}}. The template should embed the full
instructions, rubric text, and \verb{<BETTER_SAMPLE>} tagging convention.}

\item{tag_prefix}{Prefix for the better-sample tag. Defaults to
\code{"<BETTER_SAMPLE>"}.}

\item{tag_suffix}{Suffix for the better-sample tag. Defaults to
\code{"</BETTER_SAMPLE>"}.}

\item{api_key}{Optional Anthropic API key. Defaults to
\code{Sys.getenv("ANTHROPIC_API_KEY")}.}

\item{anthropic_version}{Anthropic API version string passed as the
\code{anthropic-version} HTTP header. Defaults to \code{"2023-06-01"}.}

\item{reasoning}{Character scalar indicating whether to allow more extensive
internal "thinking" before the visible answer. Two values are recognised:
\itemize{
\item \code{"none"} – standard prompting (recommended default).
\item \code{"enabled"} – uses Anthropic's extended thinking mode by
sending a \code{thinking} block with a token budget; this also changes
the default \code{max_tokens} and constrains \code{temperature}.
}}

\item{include_raw}{Logical; if \code{TRUE}, adds a list-column
\code{raw_response} containing the parsed JSON body returned by Anthropic
(or \code{NULL} on parse failure). This is useful for debugging parsing
problems.}

\item{include_thoughts}{Logical or \code{NULL}. When \code{TRUE} and
\code{reasoning = "none"}, this function upgrades to extended thinking
mode by setting \code{reasoning = "enabled"} before constructing the
request, which in turn implies \code{temperature = 1} and adds a
\code{thinking} block. When \code{FALSE} and \code{reasoning = "enabled"},
a warning is issued but extended thinking is still used. When
\code{NULL} (the default), \code{reasoning} is used as-is.}

\item{...}{Additional Anthropic parameters such as \code{max_tokens},
\code{temperature}, \code{top_p} or a custom \code{thinking_budget_tokens},
which will be passed through to the Messages API.

When \code{reasoning = "none"} the defaults are:
\itemize{
\item \code{temperature = 0} (deterministic behaviour) unless you
supply \code{temperature} explicitly.
\item \code{max_tokens = 768} unless you supply \code{max_tokens}.
}

When \code{reasoning = "enabled"} (extended thinking), the Anthropic API
imposes additional constraints:
\itemize{
\item \code{temperature} \strong{must} be 1. If you supply a different
value, this function will throw an error.
\item \code{thinking_budget_tokens} must satisfy
\code{thinking_budget_tokens >= 1024} and
\code{thinking_budget_tokens < max_tokens}. If you supply a value that
violates these constraints, this function will throw an error.
\item By default, \code{max_tokens = 2048} and
\code{thinking_budget_tokens = 1024}.
}}
}
\value{
A tibble with one row and columns:
\describe{
\item{custom_id}{ID string of the form \code{"LIVE_<ID1>_vs_<ID2>"}.}
\item{ID1, ID2}{The sample IDs you supplied.}
\item{model}{Model name reported by the API.}
\item{object_type}{Anthropic object type (for example \code{"message"}).}
\item{status_code}{HTTP-style status code (200 if successful).}
\item{error_message}{Error message if something goes wrong; otherwise NA.}
\item{thoughts}{Summarised thinking / reasoning text when
\code{reasoning = "enabled"} and the API returns thinking blocks;
otherwise \code{NA}.}
\item{content}{Concatenated text from the assistant output (excluding
thinking blocks).}
\item{better_sample}{"SAMPLE_1", "SAMPLE_2", or NA.}
\item{better_id}{ID1 if SAMPLE_1 is chosen, ID2 if SAMPLE_2 is chosen,
otherwise NA.}
\item{prompt_tokens}{Prompt / input token count (if reported).}
\item{completion_tokens}{Completion / output token count (if reported).}
\item{total_tokens}{Total token count (reported by the API or computed as
input + output tokens when not provided).}
\item{raw_response}{(Optional) list-column containing the parsed JSON
body.}
}
}
\description{
This function sends a single pairwise comparison prompt to the Anthropic
Messages API (Claude models) and parses the result into a small tibble.
}
\details{
It mirrors the behaviour and output schema of
\code{\link{openai_compare_pair_live}}, but targets Anthropic's
\code{/v1/messages} endpoint. The prompt template, \verb{<BETTER_SAMPLE>} tag
convention, and downstream parsing / BT modelling can remain unchanged.

The function is designed to work with Claude models such as Sonnet, Haiku,
and Opus in the "4.5" family. You can pass any valid Anthropic model string,
for example:
\itemize{
\item \code{"claude-sonnet-4-5"}
\item \code{"claude-haiku-4-5"}
\item \code{"claude-opus-4-5"}
}
The API typically responds with a dated model string such as
\code{"claude-sonnet-4-5-20250929"} in the \code{model} field.

\strong{Recommended defaults for pairwise writing comparisons}

For stable, reproducible comparisons we recommend:
\itemize{
\item \code{reasoning = "none"} with \code{temperature = 0} and
\code{max_tokens = 768} for standard pairwise scoring.
\item \code{reasoning = "enabled"} when you explicitly want extended
thinking; in this mode Anthropic requires \code{temperature = 1}.
The default in this function is \code{max_tokens = 2048} and
\code{thinking_budget_tokens = 1024}, which satisfies the documented
constraints \code{thinking_budget_tokens >= 1024} and
\code{thinking_budget_tokens < max_tokens}.
}

When \code{reasoning = "enabled"}, this function also sends a
\code{thinking} block to the Anthropic API:

\preformatted{
"thinking": {
  "type": "enabled",
  "budget_tokens": <thinking_budget_tokens>
}
}

Setting \code{include_thoughts = TRUE} when \code{reasoning = "none"}
is a convenient way to opt into Anthropic's extended thinking mode without
changing the \code{reasoning} argument explicitly. In that case,
\code{reasoning} is upgraded to \code{"enabled"}, the default
\code{temperature} becomes 1, and a \code{thinking} block is included in the
request. When \code{reasoning = "none"} and \code{include_thoughts} is
\code{FALSE} or \code{NULL}, the default temperature remains 0 unless
you explicitly override it.
}
\examples{
\dontrun{
# Requires ANTHROPIC_API_KEY and network access.
library(pairwiseLLM)

data("example_writing_samples", package = "pairwiseLLM")
samples <- example_writing_samples[1:2, ]

td <- trait_description("overall_quality")
tmpl <- set_prompt_template()

# Short, deterministic comparison with no explicit thinking block
res_claude <- anthropic_compare_pair_live(
  ID1               = samples$ID[1],
  text1             = samples$text[1],
  ID2               = samples$ID[2],
  text2             = samples$text[2],
  model             = "claude-sonnet-4-5",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  reasoning         = "none",
  include_raw       = FALSE
)

res_claude$better_id

# Allow more internal thinking and a longer explanation
res_claude_reason <- anthropic_compare_pair_live(
  ID1               = samples$ID[1],
  text1             = samples$text[1],
  ID2               = samples$ID[2],
  text2             = samples$text[2],
  model             = "claude-sonnet-4-5",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  reasoning         = "enabled",
  include_raw       = TRUE,
  include_thoughts  = TRUE
)

res_claude_reason$total_tokens
substr(res_claude_reason$content, 1, 200)
}

}
