#' Generate Synthetic Epidemic Data Using the Renewal Equation
#'
#' Simulates epidemic incidence data with known reproduction numbers using the renewal
#' equation framework. This function is useful for testing and validating reproduction
#' number estimation methods, as it generates synthetic outbreaks with ground truth
#' R values that can be compared against estimated values.
#'
#' The function implements the discrete renewal equation:
#' \deqn{\lambda_t = \sum_{s=1}^{t-1} I_s \cdot R_s \cdot w(t-s)}
#'
#' where \eqn{\lambda_t} is the expected number of new infections at time \eqn{t},
#' \eqn{I_s} is the incidence at time \eqn{s}, \eqn{R_s} is the reproduction number
#' at time \eqn{s}, and \eqn{w(t-s)} is the probability mass function of the serial
#' interval distribution for interval \eqn{t-s}.
#'
#' New cases at each time point are drawn from a Poisson distribution with mean
#' \eqn{\lambda_t}, introducing realistic stochastic variation while maintaining
#' the specified reproduction number trajectory.
#'
#' @param true_r numeric vector; the true time-varying reproduction numbers. The length
#'               of this vector determines the number of days in the simulated epidemic
#' @param si_mean numeric; the mean of the serial interval distribution in days
#' @param si_sd numeric; the standard deviation of the serial interval distribution in days
#' @param si_dist character; the distribution family for the serial interval. Must be
#'                either "gamma" (default) or "normal". Gamma is recommended as it
#'                naturally restricts to positive values
#' @param initial_cases integer; the number of cases on the first day of the epidemic.
#'                      Defaults to 10
#'
#' @return A data frame with three columns:
#' \itemize{
#'   \item \code{date}: Date sequence starting from "2023-01-01"
#'   \item \code{true_r}: The input reproduction number values
#'   \item \code{incidence}: Simulated daily case counts
#' }
#'
#' @details
#' The serial interval distribution is truncated at the 99th percentile to avoid
#' computationally expensive calculations for very long tails. For the normal
#' distribution, the probability mass function is normalized to ensure proper
#' probability weights.
#'
#' This function is particularly useful for:
#' \itemize{
#'   \item Validating reproduction number estimation methods
#'   \item Testing the performance of epidemiological models
#'   \item Generating realistic epidemic scenarios for research
#'   \item Creating training data for machine learning approaches
#' }
#'
#' @seealso \code{\link{wallinga_lipsitch}} for reproduction
#'          number estimation methods that can be applied to the generated data
#'
#' @references
#' Fraser C (2007). Estimating individual and household reproduction numbers in an
#' emerging epidemic. PLoS One, 2(8), e758.
#'
#' Cori A, Ferguson NM, Fraser C, Cauchemez S (2013). A new framework and software
#' to estimate time-varying reproduction numbers during epidemics. American Journal
#' of Epidemiology, 178(9), 1505-1512.
#'
#' @importFrom stats pgamma pnorm qgamma qnorm quantile rpois
#' @export
#' @examples
#' # Simple epidemic with constant R = 1.5
#' constant_r <- rep(1.5, 30)
#' epidemic1 <- generate_synthetic_epidemic(
#'   true_r = constant_r,
#'   si_mean = 7,
#'   si_sd = 3,
#'   si_dist = "gamma"
#' )
#' head(epidemic1)
#'
#' # Epidemic with declining R (e.g., intervention effect)
#' declining_r <- seq(2.0, 0.5, length.out = 50)
#' epidemic2 <- generate_synthetic_epidemic(
#'   true_r = declining_r,
#'   si_mean = 5,
#'   si_sd = 2,
#'   si_dist = "gamma",
#'   initial_cases = 5
#' )
#'
#' # Epidemic with seasonal pattern
#' days <- 100
#' seasonal_r <- 1.2 + 0.5 * sin(2 * pi * (1:days) / 365 * 7) # Weekly seasonality
#' epidemic3 <- generate_synthetic_epidemic(
#'   true_r = seasonal_r,
#'   si_mean = 6,
#'   si_sd = 2.5,
#'   si_dist = "normal"
#' )
#'
#' # Plot the results
#' if (require(ggplot2)) {
#'   library(ggplot2)
#'   ggplot(epidemic1, aes(x = date)) +
#'     geom_col(aes(y = incidence), alpha = 0.7) +
#'     geom_line(aes(y = true_r * 10), color = "red") +
#'     labs(title = "Synthetic Epidemic",
#'          y = "Daily Incidence",
#'          subtitle = "Red line: True R × 10")
#' }
generate_synthetic_epidemic <- function(
  true_r,
  si_mean,
  si_sd,
  si_dist = "gamma",
  initial_cases = 10
) {
  n_days <- length(true_r)
  dates <- seq(as.Date("2023-01-01"), by = "day", length.out = n_days)

  # Generate serial interval distribution
  if (si_dist == "gamma") {
    shape <- (si_mean / si_sd)^2
    rate <- si_mean / (si_sd^2)
    max_si <- qgamma(0.99, shape, rate)
    si_prob <- dgamma(1:ceiling(max_si), shape, rate)
  } else {
    max_si <- qnorm(0.99, si_mean, si_sd)
    si_prob <- dnorm(1:ceiling(max_si), si_mean, si_sd)
    si_prob <- si_prob / sum(si_prob) # Normalize to ensure sum equals 1
  }

  # Initialize incidence
  incidence <- rep(0, n_days)
  incidence[1] <- initial_cases

  # Generate cases using the renewal equation directly
  for (t in 2:n_days) {
    # Sum over all previous days, weighted by serial interval
    lambda_t <- 0
    for (s in 1:(t - 1)) {
      interval <- t - s
      if (interval <= length(si_prob)) {
        lambda_t <- lambda_t + incidence[s] * true_r[s] * si_prob[interval]
      }
    }
    # Generate new cases from Poisson distribution with expected value lambda_t
    incidence[t] <- rpois(1, lambda_t)
  }

  # Return data frame
  return(
    data.frame(
      date = dates,
      true_r = true_r,
      incidence = incidence
    )
  )
}
