% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/penultimate.R
\name{penultimate}
\alias{penultimate}
\title{Smith's penultimate approximations}
\usage{
penultimate(family, method = c("bm", "pot"), thresh, qlev, m, ...)
}
\arguments{
\item{family}{the name of the parametric family. Will be used to obtain \code{dfamily}, \code{pfamily}, \code{qfamily}}

\item{method}{either block maxima (\code{'bm'}) or peaks-over-threshold (\code{'pot'}) are supported}

\item{thresh}{vector of thresholds for method \code{'pot'}}

\item{qlev}{vector of quantile levels for method \code{'pot'}, e.g., 0.9, 0.95, ... Ignored if argument \code{thresh} is provided.}

\item{m}{vector of block sizes for method \code{'bm'}}

\item{...}{additional arguments passed to \code{densF} and \code{distF}}
}
\value{
a data frame containing
\itemize{
\item \code{loc}: location parameters (\code{method='bm'})
\item \code{scale}: scale parameters
\item \code{shape}: shape parameters
\item \code{thresh}: thresholds (if \code{method='pot'}), percentile corresponding to threshold (if \code{method='pot'})
\item \code{m}: block sizes (if \code{method='bm'})
}
}
\description{
The function takes as arguments the distribution and density functions. There are two options:
\code{method='bm'} yields block maxima and \code{method='pot'} threshold exceedances.
For \code{method='bm'}, the user should provide in such case the block sizes via the
argument \code{m}, whereas if \code{method='pot'}, a vector of threshold values should be
provided. The other argument (\code{thresh} or \code{m} depending on the method) is ignored.
}
\details{
Alternatively, the user can provide functions \code{densF}, \code{quantF} and \code{distF} for the density,
quantile function and distribution functions, respectively. The user can also supply the derivative
of the density function, \code{ddensF}. If the latter is missing, it will be approximated using finite-differences.

For \code{method = "pot"}, the function computes the reciprocal hazard and its derivative on the log scale to avoid numerical overflow. Thus, the density function should have argument \code{log} and the distribution function arguments \code{log.p} and \code{lower.tail}, respectively.
}
\examples{
# Threshold exceedance for Normal variables
quants <- seq(1, 5, by = 0.02)
penult <- penultimate(
   family = "norm",
   method = 'pot',
   thresh = quants,
   ddensF = function(x){-x*dnorm(x)}, # optional argument
   )
plot(x = quants,
     y = penult$shape,
     type = 'l',
     xlab = 'quantile',
    ylab = 'Penultimate shape',
    ylim = c(-0.5, 0))
# Block maxima for Gamma variables
# User must provide arguments for shape (or rate), for which there is no default
m <- seq(30, 3650, by = 30)
penult <- penultimate(family = 'gamma', method = 'bm', m = m, shape = 0.1)
plot(x = m,
     y = penult$shape,
     type = 'l',
     xlab = 'quantile',
     ylab = 'penultimate shape')

# Comparing density of GEV approximation with true density of maxima
m <- 100 # block of size 100
p <- penultimate(
  family = 'norm',
  ddensF = function(x){-x*dnorm(x)},
  method = 'bm',
  m = m)
x <- seq(1, 5, by = 0.01)
plot(
  x = x,
  y = m * dnorm(x) * exp((m-1) * pnorm(x, log.p = TRUE)),
  type = 'l',
  ylab = 'density',
  main = 'Distribution of the maxima of\n 100 standard normal variates')
lines(x, mev::dgev(x, loc = p$loc, scale = p$scale, shape = 0), col = 2)
lines(x, mev::dgev(x, loc = p$loc, scale = p$scale, shape = p$shape), col = 4)
legend(
 x = 'topright',
 lty = c(1, 1, 1),
 col = c(1, 2, 4),
 legend = c('exact', 'ultimate', 'penultimate'),
 bty = 'n')
}
\references{
Smith, R.L. (1987). Approximations in extreme value theory. \emph{Technical report 205}, Center for Stochastic Process, University of North Carolina, 1--34.
}
\author{
Leo Belzile
}
