% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metrics.R
\name{luz_metric}
\alias{luz_metric}
\title{Creates a new luz metric}
\usage{
luz_metric(
  name = NULL,
  ...,
  private = NULL,
  active = NULL,
  parent_env = parent.frame(),
  inherit = NULL
)
}
\arguments{
\item{name}{string naming the new metric.}

\item{...}{named list of public methods. You should implement at least
\code{initialize}, \code{update} and \code{compute}. See the details section for more
information.}

\item{private}{An optional list of private members, which can be functions
and non-functions.}

\item{active}{An optional list of active binding functions.}

\item{parent_env}{An environment to use as the parent of newly-created
objects.}

\item{inherit}{A R6ClassGenerator object to inherit from; in other words, a
superclass. This is captured as an unevaluated expression which is
evaluated in \code{parent_env} each time an object is instantiated.}
}
\value{
Returns new luz metric.
}
\description{
Creates a new luz metric
}
\details{
In order to implement a new \code{luz_metric} we need to implement 3 methods:
\itemize{
\item \code{initialize}: defines the metric initial state. This function is
called for each epoch for both training and validation loops.
\item \code{update}: updates the metric internal state. This function is called
at every training and validation step with the predictions obtained by
the model and the target values obtained from the dataloader.
\item \code{compute}: uses the internal state to compute metric values. This
function is called whenever we need to obtain the current metric
value. Eg, it’s called every training step for metrics displayed in
the progress bar, but only called once per epoch to record it’s value
when the progress bar is not displayed.
}

Optionally, you can implement an \code{abbrev} field that gives the metric an
abbreviation that will be used when displaying metric information in the
console or tracking record. If no \code{abbrev} is passed, the class name
will be used.

Let’s take a look at the implementation of \code{luz_metric_accuracy} so you
can see how to implement a new one:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{luz_metric_accuracy <- luz_metric(
  # An abbreviation to be shown in progress bars, or 
  # when printing progress
  abbrev = "Acc", 
  # Initial setup for the metric. Metrics are initialized
  # every epoch, for both training and validation
  initialize = function() \{
    self$correct <- 0
    self$total <- 0
  \},
  # Run at every training or validation step and updates
  # the internal state. The update function takes `preds`
  # and `target` as parameters.
  update = function(preds, target) \{
    pred <- torch::torch_argmax(preds, dim = 2)
    self$correct <- self$correct + (pred == target)$
      to(dtype = torch::torch_float())$
      sum()$
      item()
    self$total <- self$total + pred$numel()
  \},
  # Use the internal state to query the metric value
  compute = function() \{
    self$correct/self$total
  \}
)
}\if{html}{\out{</div>}}

\strong{Note}: It’s good practice that the \code{compute} metric returns regular R
values instead of torch tensors and other parts of luz will expect that.
}
\examples{
luz_metric_accuracy <- luz_metric(
  # An abbreviation to be shown in progress bars, or
  # when printing progress
  abbrev = "Acc",
  # Initial setup for the metric. Metrics are initialized
  # every epoch, for both training and validation
  initialize = function() {
    self$correct <- 0
    self$total <- 0
  },
  # Run at every training or validation step and updates
  # the internal state. The update function takes `preds`
  # and `target` as parameters.
  update = function(preds, target) {
    pred <- torch::torch_argmax(preds, dim = 2)
    self$correct <- self$correct + (pred == target)$
      to(dtype = torch::torch_float())$
      sum()$
      item()
    self$total <- self$total + pred$numel()
  },
  # Use the internal state to query the metric value
  compute = function() {
    self$correct/self$total
  }
)

}
\seealso{
Other luz_metrics: 
\code{\link{luz_metric_accuracy}()},
\code{\link{luz_metric_binary_accuracy}()},
\code{\link{luz_metric_binary_accuracy_with_logits}()},
\code{\link{luz_metric_binary_auroc}()},
\code{\link{luz_metric_mae}()},
\code{\link{luz_metric_mse}()},
\code{\link{luz_metric_multiclass_auroc}()},
\code{\link{luz_metric_rmse}()}
}
\concept{luz_metrics}
