% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distances_omv.R
\name{distances_omv}
\alias{distances_omv}
\title{Calculates distances (returning a symmetric matrix) from a raw data matrix in .omv-files for the
statistical spreadsheet 'jamovi' (\url{https://www.jamovi.org})}
\usage{
distances_omv(
  dtaInp = NULL,
  fleOut = "",
  varDst = c(),
  clmDst = TRUE,
  stdDst = "none",
  nmeDst = "euclid",
  mtxSps = FALSE,
  mtxTrL = FALSE,
  mtxDgn = TRUE,
  usePkg = c("foreign", "haven"),
  selSet = "",
  ...
)
}
\arguments{
\item{dtaInp}{Either a data frame or the name of a data file to be read (including the path,
if required; "FILENAME.ext"; default: NULL); files can be of any supported file
type, see Details below.}

\item{fleOut}{Name of the data file to be written (including the path, if required;
"FILE_OUT.omv"; default: ""); if empty, the resulting data frame is returned
instead.}

\item{varDst}{Variable (default: c()) containing a character vector with the names of the
variables for which distances are to be calculated. See Details for more
information.}

\item{clmDst}{Whether the distances shall be calculated between columns (TRUE) or rows
(FALSE; default: TRUE). See Details for more information.}

\item{stdDst}{Character string indicating whether the variables in varDst are to be standardized
and how (default: "none"). See Details for more information.}

\item{nmeDst}{Character string indicating which distance measure is to be calculated calculated
(default: "euclidean"). See Details for more information.}

\item{mtxSps}{Whether the symmetric matrix to be returned should be sparse (default: FALSE)}

\item{mtxTrL}{Whether the symmetric matrix to be returned should only contain the lower
triangular (default: FALSE)}

\item{mtxDgn}{Whether the symmetric matrix to be returned should retain the values in the main
diagonal (default: TRUE)}

\item{usePkg}{Name of the package: "foreign" or "haven" that shall be used to read SPSS, Stata,
and SAS files; "foreign" is the default (it comes with base R), but "haven" is
newer and more comprehensive.}

\item{selSet}{Name of the data set that is to be selected from the workspace (only applies when
reading .RData-files)}

\item{...}{Additional arguments passed on to methods; see Details below.}
}
\value{
a data frame containing a symmetric matrix (only returned if \code{fleOut} is empty)
containing the distances between the variables / columns (clmDst == TRUE) or rows
(clmDst == FALSE)
}
\description{
Calculates distances (returning a symmetric matrix) from a raw data matrix in .omv-files for the
statistical spreadsheet 'jamovi' (\url{https://www.jamovi.org})
}
\details{
\itemize{
\item \code{varDst} must a character vector containing the variables to calculated distances over. If
\code{clmDst} is set to TRUE, distances are calculated between all possible variable pairs and over
subjects / rows in the original data frame. If \code{clmDst} is set to FALSE, distances are
calculated between participants and over all variables given in \code{varDst}. If \code{clmDst} is set
to \code{TRUE}, the symmetric matrix that is returned has the size V x V (V being the number of
variables in varDst; if \code{mtxSps} is set to \code{TRUE}, the size is V - 1 x V - 1, see below); if
\code{clmDst} is set to \code{FALSE}, the symmetric matrix that is returned has the size R x R (R being
the number of rows in the original dataset; it is if \code{mtxSps} is set to \code{TRUE}, the size is
R - 1 x R - 1, see below).
\item \code{stdDst} can be one of the following calculations to standardize the selected variables before
calculating the distances: \code{none} (do not standardize; default), \code{z} (z scores), \code{sd} (divide
by the std. dev.), \code{range} (divide by the range), \code{max} (divide by the absolute maximum),
\code{mean} (divide by the mean), \code{rescale} (subtract the mean and divide by the range).
\item \code{nmeDst} can be one of the following distance measures.
(1) For interval data: \code{euclid} (Euclidean), \code{seuclid} (squared Euclidean), \code{block} (city
block / Manhattan), \code{canberra} (Canberra). \code{chebychev} (maximum distance / supremum norm /
Chebychev), \code{minkowski_p} (Minkowski with power p; NB: needs p), \code{power_p_r} (Minkowski with
power p, and the r-th root; NB: needs p and r), \code{cosine} (cosine between the two vectors),
\code{correlation} (correlation between the two vectors).
(2) For frequency count data: \code{chisq} (chi-square dissimilarity between two sets of
frequencies), \code{ph2} (chi-square dissimilarity normalized by the square root of the number
of values used in the calculation).
(3) For binary data, all measure have to optional parts \code{p} and \code{np} which indicate presence
(\code{p}; defaults to 1 if not given) or absence (\code{np}; defaults to zero if not given).
(a) matching coefficients: \code{rr_p_np} (Russell and Rao), \code{sm_p_np} (simple matching),
\code{jaccard_p_np} / \code{jaccards_p_np} (Jaccard similarity; as in SPSS), \code{jaccardd_p_np} (Jaccard
dissimiliarity; as in \code{dist(..., "binary")} in R), \code{dice_p_np} (Dice or Czekanowski or
Sorenson similarity), \code{ss1_p_np} (Sokal and Sneath measure 1), \code{rt_p_np} (Rogers and
Tanimoto), \code{ss2_p_np} (Sokal and Sneath measure 2), \code{k1_p_np} (Kulczynski measure 1),
\code{ss3_p_np} (Sokal and Sneath measure 3).
(b) conditional probabilities: \code{k2_p_np} (Kulczynski measure 2), \code{ss4_p_np} (Sokal and Sneath
measure 4), \code{hamann_p_np} (Hamann).
(c) predictability measures: \code{lambda_p_np} (Goodman and Kruskal Lambda), \code{d_p_np} (Anderberg’s
D), \code{y_p_np} (Yule’s Y coefficient of colligation), \code{q_p_np} (Yule’s Q).
(d) other measures: \code{ochiai_p_np} (Ochiai), \code{ss5_p_np} (Sokal and Sneath measure 5),
\code{phi_p_np} (fourfold point correlation), \code{beuclid_p_np} (binary Euclidean distance),
\code{bseuclid_p_np} (binary squared Euclidean distance), \code{size_p_np} (size difference),
\code{pattern_p_np} (pattern difference), \code{bshape_p_np} (binary Shape difference), \code{disper_p_np}
(dispersion similarity), \code{variance_p_np} (variance dissimilarity), \code{blwmn_p_np} (binary Lance
and Williams non-metric dissimilarity).
(4) \code{none} (only carry out standardization, if stdDst is different from \code{none}).
\item If \code{mtxSps} is set, a sparse matrix is returned. Those matrices are similar to the format one
often finds for correlation matrices. The values are only retained in the lower triangular,
the columns range from the first to the variable that is second to the last in \code{varDst} (or
respectively, the columns contain the first to the second to the last row of the original
dataset when \code{clmDst} is set to \code{FALSE}), and the rows contain the second to the last variable
in \code{varDst} (or respectively, the rows contain the second to the last row of the original
dataset when \code{clmDst} is set to \code{FALSE}).
\item By default, a full symmetric matrix is returned (i.e., a matrix that has no NAs in any cell).
This behaviour can be changed with setting \code{mtxTrL} and \code{mtxDgn}: If \code{mtxTrL} is set to
\code{TRUE}, the values from the upper triangular matrix are removed / replaced with NAs; if
\code{mtxDgn} is set to \code{FALSE}, the values from the main diagonal are removed / replaced with NAs.
\item The ellipsis-parameter (\code{...}) can be used to submit arguments / parameters to the functions
that are used for reading and writing the data. By clicking on the respective function under
“See also”, you can get a more detailed overview over which parameters each of those functions
take. The functions are: \code{read_omv} and \code{write_omv} (for jamovi-files), \code{read.table} (for CSV
/ TSV files; using similar defaults as \code{read.csv} for CSV and \code{read.delim} for TSV which both
are based upon \code{read.table}), \code{load} (for .RData-files), \code{readRDS} (for .rds-files),
\code{read_sav} (needs the R-package \code{haven}) or \code{read.spss} (needs the R-package \code{foreign}) for
SPSS-files, \code{read_dta} (\code{haven}) / \code{read.dta} (\code{foreign}) for Stata-files, \code{read_sas}
(\code{haven}) for SAS-data-files, and \code{read_xpt} (\code{haven}) / \code{read.xport} (\code{foreign}) for
SAS-transport-files. If you would like to use \code{haven}, you may need to install it using
\code{install.packages("haven", dep = TRUE)}.
}
}
\examples{
# create matrices for the different types of distance measures: continuous
# (cntFrm), frequency counts (frqFrm) or binary (binFrm); all 20 R x 5 C
set.seed(1)
cntFrm <- stats::setNames(as.data.frame(matrix(rnorm(100, sd = 10),
            ncol = 5)), sprintf("C_\%02d", seq(5)))
frqFrm <- stats::setNames(as.data.frame(matrix(sample(seq(10), 100,
            replace = TRUE), ncol = 5)), sprintf("F_\%02d", seq(5)))
binFrm <- stats::setNames(as.data.frame(matrix(sample(c(TRUE, FALSE), 100,
            replace = TRUE), ncol = 5)), sprintf("B_\%02d", seq(5)))
nmeOut <- tempfile(fileext = ".omv")

# calculates the distances between columns, nmeDst is not required: "euclid"
# is the default
jmvReadWrite::distances_omv(dtaInp = cntFrm, fleOut = nmeOut, varDst =
  names(cntFrm), nmeDst = "euclid")
dtaFrm <- jmvReadWrite::read_omv(nmeOut)
unlink(nmeOut)
# the resulting matrix (10 x 10) with the Euclidian distances
print(dtaFrm)

# calculates the (Euclidean) distances between rows (clmDst = FALSE)
jmvReadWrite::distances_omv(dtaInp = cntFrm, fleOut = nmeOut, varDst =
  names(cntFrm), clmDst = FALSE, nmeDst = "euclid")
dtaFrm <- jmvReadWrite::read_omv(nmeOut)
unlink(nmeOut)
# the resulting matrix (20 x 20) with the Euclidian distances
print(dtaFrm)

# calculates the (Euclidean) distances between columns; the original data
# are z-standardized before calculating the distances (stdDst = "z")
jmvReadWrite::distances_omv(dtaInp = cntFrm, fleOut = nmeOut, varDst =
  names(cntFrm), stdDst = "z", nmeDst = "euclid")
dtaFrm <- jmvReadWrite::read_omv(nmeOut)
unlink(nmeOut)
# the resulting matrix (10 x 10) with the Euclidian distances using the
# z-standardized data
print(dtaFrm)

# calculates the correlations between columns
jmvReadWrite::distances_omv(dtaInp = cntFrm, fleOut = nmeOut, varDst =
  names(cntFrm), nmeDst = "correlation")
dtaFrm <- jmvReadWrite::read_omv(nmeOut)
unlink(nmeOut)
# the resulting matrix (10 x 10) with the correlations
print(dtaFrm)

# calculates the chi-square dissimilarity (nmeDst = "chisq") between columns
jmvReadWrite::distances_omv(dtaInp = frqFrm, fleOut = nmeOut, varDst =
  names(frqFrm), nmeDst = "chisq")
dtaFrm <- jmvReadWrite::read_omv(nmeOut)
unlink(nmeOut)
# the resulting matrix (10 x 10) with the chi-square dissimilarities
print(dtaFrm)

# calculates the Jaccard similarity (nmeDst = "jaccard") between columns
jmvReadWrite::distances_omv(dtaInp = binFrm, fleOut = nmeOut, varDst =
  names(binFrm), nmeDst = "jaccard")
dtaFrm <- jmvReadWrite::read_omv(nmeOut)
unlink(nmeOut)
# the resulting matrix (10 x 10) with the Jaccard similarities
print(dtaFrm)


}
\seealso{
\code{distances_omv} internally uses the following function for calculating the distances
for interval data \code{\link[stats:dist]{stats::dist()}}. It furthermore uses the following functions for reading
and writing data files in different formats: \code{\link[=read_omv]{read_omv()}} and
\code{\link[=write_omv]{write_omv()}} for jamovi-files, \code{\link[utils:read.table]{utils::read.table()}} for CSV / TSV files,
\code{\link[=load]{load()}} for reading .RData-files, \code{\link[=readRDS]{readRDS()}} for .rds-files, \code{\link[haven:read_spss]{haven::read_sav()}} or
\code{\link[foreign:read.spss]{foreign::read.spss()}} for SPSS-files, \code{\link[haven:read_dta]{haven::read_dta()}} or \code{\link[foreign:read.dta]{foreign::read.dta()}} for
Stata-files, \code{\link[haven:read_sas]{haven::read_sas()}} for SAS-data-files, and \code{\link[haven:read_xpt]{haven::read_xpt()}} or
\code{\link[foreign:read.xport]{foreign::read.xport()}} for SAS-transport-files.
}
