#' Generic summary function for bcmeta object in jarbes
#' @param object The object generated by the bcmeta function.
#'
#' @param digits The number of significant digits printed. The default value is 3.
#' @param ... \dots
#'
#' @export
summary.bcmeta = function(object, digits = 3, ...) {
  bugs.output = object$BUGSoutput
  bugs.summary = bugs.output$summary

  summary.m = list()

  # Model specifications
  model.spec = list()
  model.spec$link = "Normal approximation"

  # Hyper-priors for corrected model and bias model
  model.spec$hyper_corrected = list(
    mean.mu = object$prior$mean.mu,
    sd.mu = object$prior$sd.mu,
    scale.sigma.between = object$prior$scale.sigma.between,
    df.scale.between = object$prior$df.scale.between
  )

  model.spec$hyper_bias = list(
    B.lower = object$prior$B.lower,
    B.upper = object$prior$B.upper,
    a.0 = object$prior$a.0,
    a.1 = object$prior$a.1,
    nu = object$prior$nu
  )

  summary.m$model.specification = model.spec

  # Posterior of the model parameters

  # Model of Interest (Bias-Corrected)
  param_corrected_names = c("mu[1]", "mu.new", "tau")
  summary.m$model_corrected = bugs.summary[param_corrected_names, , drop = FALSE]

  row.names(summary.m$model_corrected) = c("Mean (corrected)",
                                           "Predictive effect (corrected)",
                                           "Tau (between studies sd)")

  # Model of Bias
  param_bias_names = c("mu[2]", "p.bias[2]")
  summary.m$model_biased = bugs.summary[param_bias_names, , drop = FALSE]

  row.names(summary.m$model_biased) = c("Mean bias", "Prob. biased class")

  # DIC and MCMC setup
  summary.m$DIC = bugs.output$DIC
  summary.m$pD = bugs.output$pD

  mcmc.setup = list(
    n.chains = bugs.output$n.chains,
    n.iter = bugs.output$n.iter,
    n.burnin = bugs.output$n.burnin
  )
  summary.m$mcmc.setup = mcmc.setup

  class(summary.m) = "summary.bcmeta"

  print(summary.m, digits, ...)
}

#' @export
print.summary.bcmeta = function(x, digits, ...) {
  cat('Model specifications:\n')
  model.spec = x$model.specification

  cat(paste('  Link function: ', model.spec$link, sep = ''))
  cat('\n')
  cat('\n')

  cat('  Hyper-priors parameters for the Model of Interest (Bias-Corrected):\n')
  hyper_corrected = model.spec$hyper_corrected
  cat(paste('  Prior for mu: Normal', '[', hyper_corrected$mean.mu,', ' ,hyper_corrected$sd.mu^2,']', sep = ''))
  cat('\n')
  cat(paste('  Prior for 1/tau^2: Scale.Gamma', '[', hyper_corrected$scale.sigma.between,', ' ,
            hyper_corrected$df.scale.between,']', sep = ''))
  cat('\n')

  cat('\n')
  cat('  Hyper-priors parameters for the Model of Bias:\n')
  hyper_bias = model.spec$hyper_bias
  cat(paste('  Prior bias interval: Uniform', '[', hyper_bias$B.lower,', ' ,hyper_bias$B.upper,']', sep = ''))
  cat('\n')
  cat(paste('  Prior bias probability: Beta', '[', hyper_bias$a.0,', ' ,hyper_bias$a.1,']', sep = ''))
  cat('\n')
  cat(paste('  Prior nu: ', hyper_bias$nu, sep = ''))
  cat('\n')

  cat('\n')
  cat('Posterior distributions:\n')
  cat('-------------------\n')
  cat('Model of Interest (Bias-Corrected):\n')
  print(round(x$model_corrected, digits))

  cat('\n-------------------\n')
  cat('Model of Bias:\n')
  print(round(x$model_biased, digits))

  cat('\n-------------------\n')

  mcmc = x$mcmc.setup
  cat(paste('MCMC setup (fit using jags): ', mcmc$n.chains, ' chains, each with ', mcmc$n.iter, ' iterations (first ', mcmc$n.burnin, ' discarded)', sep = ''))
  cat('\n')
  cat(paste('DIC: ', round(x$DIC, digits), sep = ''))
  cat('\n')
  cat(paste('pD: ', round(x$pD, digits), sep = ''))
  cat('\n')
}
