/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FAmodule.h" 
#define H5FA_TESTING

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FApkg.h"     
#include "H5FLprivate.h" 
#include "H5VMprivate.h" 

#define H5FA__TEST_BOGUS_VAL 42

typedef struct H5FA__test_ctx_t {
    uint32_t bogus; 
} H5FA__test_ctx_t;

static void  *H5FA__test_crt_context(void *udata);
static herr_t H5FA__test_dst_context(void *ctx);
static herr_t H5FA__test_fill(void *nat_blk, size_t nelmts);
static herr_t H5FA__test_encode(void *raw, const void *elmt, size_t nelmts, void *ctx);
static herr_t H5FA__test_decode(const void *raw, void *elmt, size_t nelmts, void *ctx);
static herr_t H5FA__test_debug(FILE *stream, int indent, int fwidth, hsize_t idx, const void *elmt);
static void  *H5FA__test_crt_dbg_context(H5F_t *f, haddr_t obj_addr);

const H5FA_class_t H5FA_CLS_TEST[1] = {{
    H5FA_CLS_TEST_ID,           
    "Testing",                  
    sizeof(uint64_t),           
    H5FA__test_crt_context,     
    H5FA__test_dst_context,     
    H5FA__test_fill,            
    H5FA__test_encode,          
    H5FA__test_decode,          
    H5FA__test_debug,           
    H5FA__test_crt_dbg_context, 
    H5FA__test_dst_context      
}};

H5FL_DEFINE_STATIC(H5FA__test_ctx_t);

static void *
H5FA__test_crt_context(void H5_ATTR_UNUSED *udata)
{
    H5FA__test_ctx_t *ctx; 
    void             *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    if (NULL == (ctx = H5FL_MALLOC(H5FA__test_ctx_t)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL, "can't allocate fixed array client callback context");

    
    ctx->bogus = H5FA__TEST_BOGUS_VAL;

    
    ret_value = ctx;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FA__test_dst_context(void *_ctx)
{
    H5FA__test_ctx_t *ctx = (H5FA__test_ctx_t *)_ctx; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(H5FA__TEST_BOGUS_VAL == ctx->bogus);

    
    ctx = H5FL_FREE(H5FA__test_ctx_t, ctx);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FA__test_fill(void *nat_blk, size_t nelmts)
{
    uint64_t fill_val = H5FA_TEST_FILL; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(nat_blk);
    assert(nelmts);

    H5VM_array_fill(nat_blk, &fill_val, sizeof(uint64_t), nelmts);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FA__test_encode(void *raw, const void *_elmt, size_t nelmts, void H5_ATTR_UNUSED *_ctx)
{
#ifndef NDEBUG
    H5FA__test_ctx_t *ctx = (H5FA__test_ctx_t *)_ctx; 
#endif
    const uint64_t *elmt = (const uint64_t *)_elmt; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(raw);
    assert(elmt);
    assert(nelmts);
    assert(H5FA__TEST_BOGUS_VAL == ctx->bogus);

    
    while (nelmts) {
        
        
        UINT64ENCODE(raw, *elmt);

        
        elmt++;

        
        nelmts--;
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FA__test_decode(const void *_raw, void *_elmt, size_t nelmts, void H5_ATTR_UNUSED *_ctx)
{
#ifndef NDEBUG
    H5FA__test_ctx_t *ctx = (H5FA__test_ctx_t *)_ctx; 
#endif
    uint64_t      *elmt = (uint64_t *)_elmt;     
    const uint8_t *raw  = (const uint8_t *)_raw; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(raw);
    assert(elmt);
    assert(nelmts);
    assert(H5FA__TEST_BOGUS_VAL == ctx->bogus);

    
    while (nelmts) {
        
        
        UINT64DECODE(raw, *elmt);

        
        elmt++;

        
        nelmts--;
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FA__test_debug(FILE *stream, int indent, int fwidth, hsize_t idx, const void *elmt)
{
    char temp_str[128]; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(stream);
    assert(elmt);

    
    snprintf(temp_str, sizeof(temp_str), "Element #%llu:", (unsigned long long)idx);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, temp_str,
            (unsigned long long)*(const uint64_t *)elmt);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5FA__test_crt_dbg_context(H5F_t H5_ATTR_UNUSED *f, haddr_t H5_ATTR_UNUSED obj_addr)
{
    H5FA__test_ctx_t *ctx; 
    void             *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    if (NULL == (ctx = H5FL_MALLOC(H5FA__test_ctx_t)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL, "can't allocate fixed array client callback context");

    
    ctx->bogus = H5FA__TEST_BOGUS_VAL;

    
    ret_value = ctx;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__get_cparam_test(const H5FA_t *fa, H5FA_create_t *cparam)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(fa);
    assert(cparam);

    
    cparam->raw_elmt_size = fa->hdr->cparam.raw_elmt_size;
    cparam->nelmts        = fa->hdr->cparam.nelmts;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

int
H5FA__cmp_cparam_test(const H5FA_create_t *cparam1, const H5FA_create_t *cparam2)
{
    int ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(cparam1);
    assert(cparam2);

    
    if (cparam1->raw_elmt_size < cparam2->raw_elmt_size)
        ret_value = -1;
    else if (cparam1->raw_elmt_size > cparam2->raw_elmt_size)
        ret_value = 1;

    FUNC_LEAVE_NOAPI(ret_value)

} 
