/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FDmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FDpkg.h"     
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5PLprivate.h" 

#define H5FD_SEQ_LIST_LEN 128

#define H5FD_LOCAL_VECTOR_LEN 8

#define H5FD_LOCAL_SEL_ARR_LEN 8

typedef struct H5FD_srt_tmp_t {
    haddr_t addr;
    size_t  index;
} H5FD_srt_tmp_t;

typedef struct H5FD_get_driver_ud_t {
    
    H5PL_vfd_key_t key;

    
    hid_t found_id; 
} H5FD_get_driver_ud_t;

static int    H5FD__get_driver_cb(void *obj, hid_t id, void *_op_data);
static herr_t H5FD__read_selection_translate(uint32_t skip_vector_cb, H5FD_t *file, H5FD_mem_t type,
                                             hid_t dxpl_id, uint32_t count, H5S_t **mem_spaces,
                                             H5S_t **file_spaces, haddr_t offsets[], size_t element_sizes[],
                                             void *bufs[] );
static herr_t H5FD__write_selection_translate(uint32_t skip_vector_cb, H5FD_t *file, H5FD_mem_t type,
                                              hid_t dxpl_id, uint32_t count, H5S_t **mem_spaces,
                                              H5S_t **file_spaces, haddr_t offsets[], size_t element_sizes[],
                                              const void *bufs[]);

H5FL_EXTERN(H5S_sel_iter_t);

herr_t
H5FD_locate_signature(H5FD_t *file, haddr_t *sig_addr)
{
    haddr_t  addr = HADDR_UNDEF;
    haddr_t  eoa  = HADDR_UNDEF;
    haddr_t  eof  = HADDR_UNDEF;
    uint8_t  buf[H5F_SIGNATURE_LEN];
    unsigned n;
    unsigned maxpow;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(file);
    assert(sig_addr);

    
    eof  = H5FD_get_eof(file, H5FD_MEM_SUPER);
    eoa  = H5FD_get_eoa(file, H5FD_MEM_SUPER);
    addr = MAX(eof, eoa);
    if (HADDR_UNDEF == addr)
        HGOTO_ERROR(H5E_IO, H5E_CANTINIT, FAIL, "unable to obtain EOF/EOA value");
    for (maxpow = 0; addr; maxpow++)
        addr >>= 1;
    maxpow = MAX(maxpow, 9);

    
    for (n = 8; n < maxpow; n++) {
        addr = (8 == n) ? 0 : (haddr_t)1 << n;
        if (H5FD_set_eoa(file, H5FD_MEM_SUPER, addr + H5F_SIGNATURE_LEN) < 0)
            HGOTO_ERROR(H5E_IO, H5E_CANTINIT, FAIL, "unable to set EOA value for file signature");
        if (H5FD_read(file, H5FD_MEM_SUPER, addr, (size_t)H5F_SIGNATURE_LEN, buf) < 0)
            HGOTO_ERROR(H5E_IO, H5E_CANTINIT, FAIL, "unable to read file signature");
        if (!memcmp(buf, H5F_SIGNATURE, (size_t)H5F_SIGNATURE_LEN))
            break;
    }

    
    if (n >= maxpow) {
        if (H5FD_set_eoa(file, H5FD_MEM_SUPER, eoa) < 0)
            HGOTO_ERROR(H5E_IO, H5E_CANTINIT, FAIL, "unable to reset EOA value");
        *sig_addr = HADDR_UNDEF;
    }
    else
        
        *sig_addr = addr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_read(H5FD_t *file, H5FD_mem_t type, haddr_t addr, size_t size, void *buf )
{
    hid_t    dxpl_id = H5I_INVALID_HID; 
    uint32_t actual_selection_io_mode;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert(buf);

    
    dxpl_id = H5CX_get_dxpl();

#ifndef H5_HAVE_PARALLEL
    
    if (0 == size)
        HGOTO_DONE(SUCCEED);
#endif 

    
    if (!(file->access_flags & H5F_ACC_SWMR_READ)) {
        haddr_t eoa;

        
        H5_BEFORE_USER_CB(FAIL)
            {
                eoa = (file->cls->get_eoa)(file, type);
            }
        H5_AFTER_USER_CB(FAIL)
        if (!H5_addr_defined(eoa))
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver get_eoa request failed");

        if ((addr + file->base_addr + size) > eoa)
            HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL, "addr overflow, addr = %llu, size = %llu, eoa = %llu",
                        (unsigned long long)(addr + file->base_addr), (unsigned long long)size,
                        (unsigned long long)eoa);
    }

    
    H5_BEFORE_USER_CB(FAIL)
        {
            
            ret_value = (file->cls->read)(file, type, dxpl_id, addr + file->base_addr, size, buf);
        }
    H5_AFTER_USER_CB(FAIL)
    if (ret_value < 0)
        HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "driver read request failed");

    
    if (type == H5FD_MEM_DRAW) {
        H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
        actual_selection_io_mode |= H5D_SCALAR_IO;
        H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_write(H5FD_t *file, H5FD_mem_t type, haddr_t addr, size_t size, const void *buf)
{
    hid_t    dxpl_id;           
    haddr_t  eoa = HADDR_UNDEF; 
    uint32_t actual_selection_io_mode;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert(buf);

    
    dxpl_id = H5CX_get_dxpl();

#ifndef H5_HAVE_PARALLEL
    
    if (0 == size)
        HGOTO_DONE(SUCCEED);
#endif 

    
    H5_BEFORE_USER_CB(FAIL)
        {
            eoa = (file->cls->get_eoa)(file, type);
        }
    H5_AFTER_USER_CB(FAIL)
    if (!H5_addr_defined(eoa))
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver get_eoa request failed");
    if ((addr + file->base_addr + size) > eoa)
        HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL, "addr overflow, addr = %llu, size=%llu, eoa=%llu",
                    (unsigned long long)(addr + file->base_addr), (unsigned long long)size,
                    (unsigned long long)eoa);

    
    H5_BEFORE_USER_CB(FAIL)
        {
            
            ret_value = (file->cls->write)(file, type, dxpl_id, addr + file->base_addr, size, buf);
        }
    H5_AFTER_USER_CB(FAIL)
    if (ret_value < 0)
        HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "driver write request failed");

    
    if (type == H5FD_MEM_DRAW) {
        H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
        actual_selection_io_mode |= H5D_SCALAR_IO;
        H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_read_vector(H5FD_t *file, uint32_t count, H5FD_mem_t types[], haddr_t addrs[], size_t sizes[],
                 void *bufs[] )
{
    bool       addrs_cooked = false;
    bool       extend_sizes = false;
    bool       extend_types = false;
    uint32_t   i;
    size_t     size      = 0;
    H5FD_mem_t type      = H5FD_MEM_DEFAULT;
    hid_t      dxpl_id   = H5I_INVALID_HID; 
    bool       is_raw    = false;           
    herr_t     ret_value = SUCCEED;         

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert((types) || (count == 0));
    assert((addrs) || (count == 0));
    assert((sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (sizes[0] != 0));
    assert((count == 0) || (types[0] != H5FD_MEM_NOLIST));

    
    dxpl_id = H5CX_get_dxpl();

#ifndef H5_HAVE_PARALLEL
    
    if (0 == count)
        HGOTO_DONE(SUCCEED);
#endif 

    if (file->base_addr > 0) {
        
        for (i = 0; i < count; i++)
            addrs[i] += file->base_addr;
        addrs_cooked = true;
    }

    
    if ((!(file->access_flags & H5F_ACC_SWMR_READ)) && (count > 0)) {
        haddr_t eoa;

        extend_sizes = false;
        extend_types = false;

        for (i = 0; i < count; i++) {
            if (!extend_sizes) {
                if (sizes[i] == 0) {
                    extend_sizes = true;
                    size         = sizes[i - 1];
                }
                else
                    size = sizes[i];
            }

            if (!extend_types) {
                if (types[i] == H5FD_MEM_NOLIST) {
                    extend_types = true;
                    type         = types[i - 1];
                }
                else {
                    type = types[i];

                    
                    if (type == H5FD_MEM_DRAW)
                        is_raw = true;
                }
            }

            
            H5_BEFORE_USER_CB(FAIL)
                {
                    eoa = (file->cls->get_eoa)(file, type);
                }
            H5_AFTER_USER_CB(FAIL)
            if (!H5_addr_defined(eoa))
                HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver get_eoa request failed");

            if ((addrs[i] + size) > eoa)
                HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL,
                            "addr overflow, addrs[%d] = %llu, sizes[%d] = %llu, eoa = %llu", (int)i,
                            (unsigned long long)(addrs[i]), (int)i, (unsigned long long)size,
                            (unsigned long long)eoa);
        }
    }
    else
        
        for (i = 0; i < count && types[i] != H5FD_MEM_NOLIST; i++)
            if (types[i] == H5FD_MEM_DRAW) {
                is_raw = true;
                break;
            }

    
    if (file->cls->read_vector) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->read_vector)(file, dxpl_id, count, types, addrs, sizes, bufs);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "driver read vector request failed");

        
        if (is_raw) {
            uint32_t actual_selection_io_mode;

            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_VECTOR_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }
    else {
        
        extend_sizes = false;
        extend_types = false;
        uint32_t no_selection_io_cause;
        uint32_t actual_selection_io_mode;

        for (i = 0; i < count; i++) {
            
            if (!extend_sizes) {
                if (sizes[i] == 0) {
                    extend_sizes = true;
                    size         = sizes[i - 1];
                }
                else
                    size = sizes[i];
            }

            if (!extend_types) {
                if (types[i] == H5FD_MEM_NOLIST) {
                    extend_types = true;
                    type         = types[i - 1];
                }
                else
                    type = types[i];
            }

            
            H5_BEFORE_USER_CB(FAIL)
                {
                    ret_value = (file->cls->read)(file, type, dxpl_id, addrs[i], size, bufs[i]);
                }
            H5_AFTER_USER_CB(FAIL)
            if (ret_value < 0)
                HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "driver read request failed");
        }

        
        H5CX_get_no_selection_io_cause(&no_selection_io_cause);
        no_selection_io_cause |= H5D_SEL_IO_NO_VECTOR_OR_SELECTION_IO_CB;
        H5CX_set_no_selection_io_cause(no_selection_io_cause);

        
        if (is_raw) {
            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_SCALAR_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }

done:
    
    if (addrs_cooked) {
        assert(file->base_addr > 0);
        for (i = 0; i < count; i++)
            addrs[i] -= file->base_addr;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_write_vector(H5FD_t *file, uint32_t count, H5FD_mem_t types[], haddr_t addrs[], size_t sizes[],
                  const void *bufs[])
{
    bool       addrs_cooked = false;
    bool       extend_sizes = false;
    bool       extend_types = false;
    uint32_t   i;
    size_t     size = 0;
    H5FD_mem_t type = H5FD_MEM_DEFAULT;
    hid_t      dxpl_id;                 
    haddr_t    eoa       = HADDR_UNDEF; 
    bool       is_raw    = false;       
    herr_t     ret_value = SUCCEED;     

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert((types) || (count == 0));
    assert((addrs) || (count == 0));
    assert((sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (sizes[0] != 0));
    assert((count == 0) || (types[0] != H5FD_MEM_NOLIST));

    
    dxpl_id = H5CX_get_dxpl();

#ifndef H5_HAVE_PARALLEL
    
    if (0 == count)
        HGOTO_DONE(SUCCEED);
#endif 

    if (file->base_addr > 0) {
        
        for (i = 0; i < count; i++)
            addrs[i] += file->base_addr;
        addrs_cooked = true;
    }

    extend_sizes = false;
    extend_types = false;

    for (i = 0; i < count; i++) {
        if (!extend_sizes) {
            if (sizes[i] == 0) {
                extend_sizes = true;
                size         = sizes[i - 1];
            }
            else
                size = sizes[i];
        }

        if (!extend_types) {
            if (types[i] == H5FD_MEM_NOLIST) {
                extend_types = true;
                type         = types[i - 1];
            }
            else {
                type = types[i];

                
                if (type == H5FD_MEM_DRAW)
                    is_raw = true;
            }
        }

        
        H5_BEFORE_USER_CB(FAIL)
            {
                eoa = (file->cls->get_eoa)(file, type);
            }
        H5_AFTER_USER_CB(FAIL)
        if (!H5_addr_defined(eoa))
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver get_eoa request failed");

        if ((addrs[i] + size) > eoa)
            HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL,
                        "addr overflow, addrs[%d] = %llu, sizes[%d] = %llu, eoa = %llu", (int)i,
                        (unsigned long long)(addrs[i]), (int)i, (unsigned long long)size,
                        (unsigned long long)eoa);
    }

    
    if (file->cls->write_vector) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->write_vector)(file, dxpl_id, count, types, addrs, sizes, bufs);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "driver write vector request failed");

        
        if (is_raw) {
            uint32_t actual_selection_io_mode;

            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_VECTOR_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }
    else {
        
        extend_sizes = false;
        extend_types = false;
        uint32_t no_selection_io_cause;
        uint32_t actual_selection_io_mode;

        for (i = 0; i < count; i++) {
            
            if (!extend_sizes) {
                if (sizes[i] == 0) {
                    extend_sizes = true;
                    size         = sizes[i - 1];
                }
                else
                    size = sizes[i];
            }

            if (!extend_types) {
                if (types[i] == H5FD_MEM_NOLIST) {
                    extend_types = true;
                    type         = types[i - 1];
                }
                else
                    type = types[i];
            }

            
            H5_BEFORE_USER_CB(FAIL)
                {
                    ret_value = (file->cls->write)(file, type, dxpl_id, addrs[i], size, bufs[i]);
                }
            H5_AFTER_USER_CB(FAIL)
            if (ret_value < 0)
                HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "driver write request failed");
        }

        
        H5CX_get_no_selection_io_cause(&no_selection_io_cause);
        no_selection_io_cause |= H5D_SEL_IO_NO_VECTOR_OR_SELECTION_IO_CB;
        H5CX_set_no_selection_io_cause(no_selection_io_cause);

        
        if (is_raw) {
            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_SCALAR_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }

done:
    
    if (addrs_cooked) {
        assert(file->base_addr > 0);
        for (i = 0; i < count; i++)
            addrs[i] -= file->base_addr;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__read_selection_translate(uint32_t skip_vector_cb, H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id,
                               uint32_t count, H5S_t **mem_spaces, H5S_t **file_spaces, haddr_t offsets[],
                               size_t element_sizes[], void *bufs[] )
{
    bool            extend_sizes = false;
    bool            extend_bufs  = false;
    uint32_t        i;
    size_t          element_size = 0;
    void           *buf          = NULL;
    bool            use_vector   = false;
    haddr_t         addrs_local[H5FD_LOCAL_VECTOR_LEN];
    haddr_t        *addrs = addrs_local;
    size_t          sizes_local[H5FD_LOCAL_VECTOR_LEN];
    size_t         *sizes = sizes_local;
    void           *vec_bufs_local[H5FD_LOCAL_VECTOR_LEN];
    void          **vec_bufs = vec_bufs_local;
    hsize_t         file_off[H5FD_SEQ_LIST_LEN];
    size_t          file_len[H5FD_SEQ_LIST_LEN];
    hsize_t         mem_off[H5FD_SEQ_LIST_LEN];
    size_t          mem_len[H5FD_SEQ_LIST_LEN];
    size_t          file_seq_i;
    size_t          mem_seq_i;
    size_t          file_nseq;
    size_t          mem_nseq;
    size_t          io_len;
    size_t          nelmts;
    hssize_t        hss_nelmts;
    size_t          seq_nelem;
    H5S_sel_iter_t *file_iter      = NULL;
    H5S_sel_iter_t *mem_iter       = NULL;
    bool            file_iter_init = false;
    bool            mem_iter_init  = false;
    H5FD_mem_t      types[2]       = {type, H5FD_MEM_NOLIST};
    size_t          vec_arr_nalloc = H5FD_LOCAL_VECTOR_LEN;
    size_t          vec_arr_nused  = 0;
    herr_t          ret_value      = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(file->cls);
    assert((mem_spaces) || (count == 0));
    assert((file_spaces) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    use_vector = (file->cls->read_vector != NULL) && (!skip_vector_cb);

    if (count > 0) {
        
        assert(element_sizes[0] != 0);
        assert(bufs[0] != NULL);

        
        if (NULL == (file_iter = H5FL_MALLOC(H5S_sel_iter_t)))
            HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "couldn't allocate file selection iterator");
        if (NULL == (mem_iter = H5FL_MALLOC(H5S_sel_iter_t)))
            HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "couldn't allocate memory selection iterator");
    }

    
    for (i = 0; i < count; i++) {

        

        if (!extend_sizes) {

            if (element_sizes[i] == 0) {

                extend_sizes = true;
                element_size = element_sizes[i - 1];
            }
            else {

                element_size = element_sizes[i];
            }
        }

        if (!extend_bufs) {

            if (bufs[i] == NULL) {

                extend_bufs = true;
                buf         = bufs[i - 1];
            }
            else {

                buf = bufs[i];
            }
        }

        
        if (H5S_select_iter_init(file_iter, file_spaces[i], element_size, 0) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "can't initialize sequence list for file space");
        file_iter_init = true;
        if (H5S_select_iter_init(mem_iter, mem_spaces[i], element_size, 0) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "can't initialize sequence list for memory space");
        mem_iter_init = true;

        
        if ((hss_nelmts = (hssize_t)H5S_GET_SELECT_NPOINTS(file_spaces[i])) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTCOUNT, FAIL, "can't get number of elements selected");
        H5_CHECKED_ASSIGN(nelmts, size_t, hss_nelmts, hssize_t);

#ifndef NDEBUG
        
        {
            if ((hss_nelmts = (hssize_t)H5S_GET_SELECT_NPOINTS(mem_spaces[i])) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTCOUNT, FAIL, "can't get number of elements selected");
            assert((hssize_t)nelmts == hss_nelmts);
        }
#endif 

        
        file_seq_i = H5FD_SEQ_LIST_LEN;
        mem_seq_i  = H5FD_SEQ_LIST_LEN;
        file_nseq  = 0;
        mem_nseq   = 0;

        
        while (file_seq_i < file_nseq || nelmts > 0) {
            
            if (file_seq_i == H5FD_SEQ_LIST_LEN) {
                if (H5S_SELECT_ITER_GET_SEQ_LIST(file_iter, H5FD_SEQ_LIST_LEN, SIZE_MAX, &file_nseq,
                                                 &seq_nelem, file_off, file_len) < 0)
                    HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, FAIL, "sequence length generation failed");
                assert(file_nseq > 0);

                nelmts -= seq_nelem;
                file_seq_i = 0;
            }
            assert(file_seq_i < file_nseq);

            
            if (mem_seq_i == H5FD_SEQ_LIST_LEN) {
                if (H5S_SELECT_ITER_GET_SEQ_LIST(mem_iter, H5FD_SEQ_LIST_LEN, SIZE_MAX, &mem_nseq, &seq_nelem,
                                                 mem_off, mem_len) < 0)
                    HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, FAIL, "sequence length generation failed");
                assert(mem_nseq > 0);

                mem_seq_i = 0;
            }
            assert(mem_seq_i < mem_nseq);

            
            io_len = MIN(file_len[file_seq_i], mem_len[mem_seq_i]);

            
            if (use_vector) {
                
                if (vec_arr_nused == vec_arr_nalloc) {
                    
                    if (addrs == addrs_local) {
                        assert(sizes == sizes_local);
                        assert(vec_bufs == vec_bufs_local);

                        
                        if (NULL == (addrs = H5MM_malloc(sizeof(addrs_local) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for address list");
                        if (NULL == (sizes = H5MM_malloc(sizeof(sizes_local) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for size list");
                        if (NULL == (vec_bufs = H5MM_malloc(sizeof(vec_bufs_local) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for buffer list");

                        
                        (void)H5MM_memcpy(addrs, addrs_local, sizeof(addrs_local));
                        (void)H5MM_memcpy(sizes, sizes_local, sizeof(sizes_local));
                        (void)H5MM_memcpy(vec_bufs, vec_bufs_local, sizeof(vec_bufs_local));
                    }
                    else {
                        void *tmp_ptr;

                        
                        if (NULL == (tmp_ptr = H5MM_realloc(addrs, vec_arr_nalloc * sizeof(*addrs) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory reallocation failed for address list");
                        addrs = tmp_ptr;
                        if (NULL == (tmp_ptr = H5MM_realloc(sizes, vec_arr_nalloc * sizeof(*sizes) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory reallocation failed for size list");
                        sizes = tmp_ptr;
                        if (NULL ==
                            (tmp_ptr = H5MM_realloc(vec_bufs, vec_arr_nalloc * sizeof(*vec_bufs) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory reallocation failed for buffer list");
                        vec_bufs = tmp_ptr;
                    }

                    
                    vec_arr_nalloc *= 2;
                }

                
                addrs[vec_arr_nused]    = offsets[i] + file_off[file_seq_i];
                sizes[vec_arr_nused]    = io_len;
                vec_bufs[vec_arr_nused] = (void *)((uint8_t *)buf + mem_off[mem_seq_i]);
                vec_arr_nused++;
            }
            else {
                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        
                        ret_value = (file->cls->read)(file, type, dxpl_id, offsets[i] + file_off[file_seq_i],
                                                      io_len, (void *)((uint8_t *)buf + mem_off[mem_seq_i]));
                    }
                H5_AFTER_USER_CB(FAIL)
                if (ret_value < 0)
                    HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "driver read request failed");
            }

            
            if (io_len == file_len[file_seq_i])
                file_seq_i++;
            else {
                file_off[file_seq_i] += io_len;
                file_len[file_seq_i] -= io_len;
            }

            
            if (io_len == mem_len[mem_seq_i])
                mem_seq_i++;
            else {
                mem_off[mem_seq_i] += io_len;
                mem_len[mem_seq_i] -= io_len;
            }
        }

        
        if (mem_seq_i < mem_nseq)
            HGOTO_ERROR(H5E_INTERNAL, H5E_BADVALUE, FAIL,
                        "file selection terminated before memory selection");

        
        if (H5S_SELECT_ITER_RELEASE(file_iter) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_CANTFREE, FAIL, "can't release file selection iterator");
        file_iter_init = false;
        if (H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_CANTFREE, FAIL, "can't release memory selection iterator");
        mem_iter_init = false;
    }

    
    if (use_vector) {
        uint32_t actual_selection_io_mode;

        H5_CHECK_OVERFLOW(vec_arr_nused, size_t, uint32_t);
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->read_vector)(file, dxpl_id, (uint32_t)vec_arr_nused, types, addrs,
                                                     sizes, vec_bufs);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "driver read vector request failed");

        
        if (type == H5FD_MEM_DRAW && count > 0) {
            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_VECTOR_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }
    else if (count > 0) {
        uint32_t no_selection_io_cause;
        uint32_t actual_selection_io_mode;

        
        H5CX_get_no_selection_io_cause(&no_selection_io_cause);
        no_selection_io_cause |= H5D_SEL_IO_NO_VECTOR_OR_SELECTION_IO_CB;
        H5CX_set_no_selection_io_cause(no_selection_io_cause);

        
        if (type == H5FD_MEM_DRAW) {
            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_SCALAR_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }

done:
    
    if (file_iter) {
        if (file_iter_init && H5S_SELECT_ITER_RELEASE(file_iter) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_CANTFREE, FAIL, "can't release file selection iterator");
        file_iter = H5FL_FREE(H5S_sel_iter_t, file_iter);
    }
    if (mem_iter) {
        if (mem_iter_init && H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_CANTFREE, FAIL, "can't release memory selection iterator");
        mem_iter = H5FL_FREE(H5S_sel_iter_t, mem_iter);
    }

    
    if (use_vector) {
        if (addrs != addrs_local)
            addrs = H5MM_xfree(addrs);
        if (sizes != sizes_local)
            sizes = H5MM_xfree(sizes);
        if (vec_bufs != vec_bufs_local)
            vec_bufs = H5MM_xfree(vec_bufs);
    }

    
    assert(!addrs || addrs == addrs_local);
    assert(!sizes || sizes == sizes_local);
    assert(!vec_bufs || vec_bufs == vec_bufs_local);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_read_selection(H5FD_t *file, H5FD_mem_t type, uint32_t count, H5S_t **mem_spaces, H5S_t **file_spaces,
                    haddr_t offsets[], size_t element_sizes[], void *bufs[] )
{
    bool     offsets_cooked = false;
    hid_t    mem_space_ids_local[H5FD_LOCAL_SEL_ARR_LEN];
    hid_t   *mem_space_ids = mem_space_ids_local;
    hid_t    file_space_ids_local[H5FD_LOCAL_SEL_ARR_LEN];
    hid_t   *file_space_ids = file_space_ids_local;
    uint32_t num_spaces     = 0;
    hid_t    dxpl_id        = H5I_INVALID_HID; 
    uint32_t i;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert((mem_spaces) || (count == 0));
    assert((file_spaces) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (element_sizes[0] != 0));
    assert((count == 0) || (bufs[0] != NULL));

    
    dxpl_id = H5CX_get_dxpl();

#ifndef H5_HAVE_PARALLEL
    
    if (0 == count)
        HGOTO_DONE(SUCCEED);
#endif 

    if (file->base_addr > 0) {
        
        for (i = 0; i < count; i++)
            offsets[i] += file->base_addr;
        offsets_cooked = true;
    }

    
    
    if (!(file->access_flags & H5F_ACC_SWMR_READ)) {
        haddr_t eoa;

        
        H5_BEFORE_USER_CB(FAIL)
            {
                eoa = (file->cls->get_eoa)(file, type);
            }
        H5_AFTER_USER_CB(FAIL)
        if (!H5_addr_defined(eoa))
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver get_eoa request failed");

        for (i = 0; i < count; i++)
            if ((offsets[i]) > eoa)
                HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL, "addr overflow, offsets[%d] = %llu, eoa = %llu",
                            (int)i, (unsigned long long)(offsets[i]), (unsigned long long)eoa);
    }

    
    if (file->cls->read_selection) {
        uint32_t actual_selection_io_mode;

        
        if (count > sizeof(mem_space_ids_local) / sizeof(mem_space_ids_local[0])) {
            if (NULL == (mem_space_ids = H5MM_malloc(count * sizeof(hid_t))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed for dataspace list");
            if (NULL == (file_space_ids = H5MM_malloc(count * sizeof(hid_t))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed for dataspace list");
        }

        
        for (; num_spaces < count; num_spaces++) {
            if ((mem_space_ids[num_spaces] = H5I_register(H5I_DATASPACE, mem_spaces[num_spaces], true)) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register dataspace ID");

            if ((file_space_ids[num_spaces] = H5I_register(H5I_DATASPACE, file_spaces[num_spaces], true)) <
                0) {
                if (NULL == H5I_remove(mem_space_ids[num_spaces]))
                    HDONE_ERROR(H5E_VFL, H5E_CANTREMOVE, FAIL, "problem removing id");
                HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register dataspace ID");
            }
        }

        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->read_selection)(file, type, dxpl_id, count, mem_space_ids,
                                                        file_space_ids, offsets, element_sizes, bufs);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "driver read selection request failed");

        
        if (type == H5FD_MEM_DRAW) {
            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_SELECTION_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }
    else
        
        if (H5FD__read_selection_translate(SKIP_NO_CB, file, type, dxpl_id, count, mem_spaces, file_spaces,
                                           offsets, element_sizes, bufs) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "translation to vector or scalar read failed");

done:
    
    if (offsets_cooked) {
        assert(file->base_addr > 0);
        for (i = 0; i < count; i++)
            offsets[i] -= file->base_addr;
    }

    
    for (i = 0; i < num_spaces; i++) {
        if (NULL == H5I_remove(mem_space_ids[i]))
            HDONE_ERROR(H5E_VFL, H5E_CANTREMOVE, FAIL, "problem removing id");
        if (NULL == H5I_remove(file_space_ids[i]))
            HDONE_ERROR(H5E_VFL, H5E_CANTREMOVE, FAIL, "problem removing id");
    }
    if (mem_space_ids != mem_space_ids_local)
        mem_space_ids = H5MM_xfree(mem_space_ids);
    if (file_space_ids != file_space_ids_local)
        file_space_ids = H5MM_xfree(file_space_ids);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_read_selection_id(uint32_t skip_cb, H5FD_t *file, H5FD_mem_t type, uint32_t count, hid_t mem_space_ids[],
                       hid_t file_space_ids[], haddr_t offsets[], size_t element_sizes[],
                       void *bufs[] )
{
    bool     offsets_cooked = false;
    H5S_t   *mem_spaces_local[H5FD_LOCAL_SEL_ARR_LEN];
    H5S_t  **mem_spaces = mem_spaces_local;
    H5S_t   *file_spaces_local[H5FD_LOCAL_SEL_ARR_LEN];
    H5S_t  **file_spaces = file_spaces_local;
    hid_t    dxpl_id     = H5I_INVALID_HID; 
    uint32_t i;
    uint32_t skip_selection_cb;
    uint32_t skip_vector_cb;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert((mem_space_ids) || (count == 0));
    assert((file_space_ids) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (element_sizes[0] != 0));
    assert((count == 0) || (bufs[0] != NULL));

    
    dxpl_id = H5CX_get_dxpl();

#ifndef H5_HAVE_PARALLEL
    
    if (0 == count)
        HGOTO_DONE(SUCCEED);
#endif 

    skip_selection_cb = skip_cb & SKIP_SELECTION_CB;
    skip_vector_cb    = skip_cb & SKIP_VECTOR_CB;

    if (file->base_addr > 0) {
        
        for (i = 0; i < count; i++)
            offsets[i] += file->base_addr;
        offsets_cooked = true;
    }

    
    
    if (!(file->access_flags & H5F_ACC_SWMR_READ)) {
        haddr_t eoa;

        
        H5_BEFORE_USER_CB(FAIL)
            {
                eoa = (file->cls->get_eoa)(file, type);
            }
        H5_AFTER_USER_CB(FAIL)
        if (!H5_addr_defined(eoa))
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver get_eoa request failed");

        for (i = 0; i < count; i++)
            if ((offsets[i]) > eoa)
                HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL, "addr overflow, offsets[%d] = %llu, eoa = %llu",
                            (int)i, (unsigned long long)(offsets[i]), (unsigned long long)eoa);
    }

    
    if (!skip_selection_cb && file->cls->read_selection) {
        uint32_t actual_selection_io_mode;

        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->read_selection)(file, type, dxpl_id, count, mem_space_ids,
                                                        file_space_ids, offsets, element_sizes, bufs);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "driver read selection request failed");

        
        if (type == H5FD_MEM_DRAW) {
            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_SELECTION_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }
    else {
        

        
        if (count > sizeof(mem_spaces_local) / sizeof(mem_spaces_local[0])) {
            if (NULL == (mem_spaces = H5MM_malloc(count * sizeof(H5S_t *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed for dataspace list");
            if (NULL == (file_spaces = H5MM_malloc(count * sizeof(H5S_t *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed for dataspace list");
        }

        
        for (i = 0; i < count; i++) {
            if (NULL == (mem_spaces[i] = (H5S_t *)H5I_object_verify(mem_space_ids[i], H5I_DATASPACE)))
                HGOTO_ERROR(H5E_VFL, H5E_BADTYPE, H5I_INVALID_HID, "can't retrieve memory dataspace from ID");
            if (NULL == (file_spaces[i] = (H5S_t *)H5I_object_verify(file_space_ids[i], H5I_DATASPACE)))
                HGOTO_ERROR(H5E_VFL, H5E_BADTYPE, H5I_INVALID_HID, "can't retrieve file dataspace from ID");
        }

        

        if (H5FD__read_selection_translate(skip_vector_cb, file, type, dxpl_id, count, mem_spaces,
                                           file_spaces, offsets, element_sizes, bufs) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "translation to vector or scalar read failed");
    }

done:
    
    if (offsets_cooked) {
        assert(file->base_addr > 0);
        for (i = 0; i < count; i++)
            offsets[i] -= file->base_addr;
    }

    
    if (mem_spaces != mem_spaces_local)
        mem_spaces = H5MM_xfree(mem_spaces);
    if (file_spaces != file_spaces_local)
        file_spaces = H5MM_xfree(file_spaces);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__write_selection_translate(uint32_t skip_vector_cb, H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id,
                                uint32_t count, H5S_t **mem_spaces, H5S_t **file_spaces, haddr_t offsets[],
                                size_t element_sizes[], const void *bufs[])
{
    bool            extend_sizes = false;
    bool            extend_bufs  = false;
    uint32_t        i;
    size_t          element_size = 0;
    const void     *buf          = NULL;
    bool            use_vector   = false;
    haddr_t         addrs_local[H5FD_LOCAL_VECTOR_LEN];
    haddr_t        *addrs = addrs_local;
    size_t          sizes_local[H5FD_LOCAL_VECTOR_LEN];
    size_t         *sizes = sizes_local;
    const void     *vec_bufs_local[H5FD_LOCAL_VECTOR_LEN];
    const void    **vec_bufs = vec_bufs_local;
    hsize_t         file_off[H5FD_SEQ_LIST_LEN];
    size_t          file_len[H5FD_SEQ_LIST_LEN];
    hsize_t         mem_off[H5FD_SEQ_LIST_LEN];
    size_t          mem_len[H5FD_SEQ_LIST_LEN];
    size_t          file_seq_i;
    size_t          mem_seq_i;
    size_t          file_nseq;
    size_t          mem_nseq;
    size_t          io_len;
    size_t          nelmts;
    hssize_t        hss_nelmts;
    size_t          seq_nelem;
    H5S_sel_iter_t *file_iter      = NULL;
    H5S_sel_iter_t *mem_iter       = NULL;
    bool            file_iter_init = false;
    bool            mem_iter_init  = false;
    H5FD_mem_t      types[2]       = {type, H5FD_MEM_NOLIST};
    size_t          vec_arr_nalloc = H5FD_LOCAL_VECTOR_LEN;
    size_t          vec_arr_nused  = 0;
    herr_t          ret_value      = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(file->cls);
    assert((mem_spaces) || (count == 0));
    assert((file_spaces) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    use_vector = (file->cls->write_vector != NULL) && (!skip_vector_cb);

    if (count > 0) {
        
        assert(element_sizes[0] != 0);
        assert(bufs[0] != NULL);

        
        if (NULL == (file_iter = H5FL_MALLOC(H5S_sel_iter_t)))
            HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "couldn't allocate file selection iterator");
        if (NULL == (mem_iter = H5FL_MALLOC(H5S_sel_iter_t)))
            HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, FAIL, "couldn't allocate memory selection iterator");
    }

    
    for (i = 0; i < count; i++) {
        
        if (!extend_sizes) {
            if (element_sizes[i] == 0) {
                extend_sizes = true;
                element_size = element_sizes[i - 1];
            }
            else
                element_size = element_sizes[i];
        }

        if (!extend_bufs) {
            if (bufs[i] == NULL) {
                extend_bufs = true;
                buf         = bufs[i - 1];
            }
            else
                buf = bufs[i];
        }

        
        if (H5S_select_iter_init(file_iter, file_spaces[i], element_size, 0) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "can't initialize sequence list for file space");
        file_iter_init = true;
        if (H5S_select_iter_init(mem_iter, mem_spaces[i], element_size, 0) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "can't initialize sequence list for memory space");
        mem_iter_init = true;

        
        if ((hss_nelmts = (hssize_t)H5S_GET_SELECT_NPOINTS(file_spaces[i])) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTCOUNT, FAIL, "can't get number of elements selected");
        H5_CHECKED_ASSIGN(nelmts, size_t, hss_nelmts, hssize_t);

#ifndef NDEBUG
        
        {
            if ((hss_nelmts = (hssize_t)H5S_GET_SELECT_NPOINTS(mem_spaces[i])) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTCOUNT, FAIL, "can't get number of elements selected");
            assert((hssize_t)nelmts == hss_nelmts);
        }
#endif 

        
        file_seq_i = H5FD_SEQ_LIST_LEN;
        mem_seq_i  = H5FD_SEQ_LIST_LEN;
        file_nseq  = 0;
        mem_nseq   = 0;

        
        while (file_seq_i < file_nseq || nelmts > 0) {
            
            if (file_seq_i == H5FD_SEQ_LIST_LEN) {
                if (H5S_SELECT_ITER_GET_SEQ_LIST(file_iter, H5FD_SEQ_LIST_LEN, SIZE_MAX, &file_nseq,
                                                 &seq_nelem, file_off, file_len) < 0)
                    HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, FAIL, "sequence length generation failed");
                assert(file_nseq > 0);

                nelmts -= seq_nelem;
                file_seq_i = 0;
            }
            assert(file_seq_i < file_nseq);

            
            if (mem_seq_i == H5FD_SEQ_LIST_LEN) {
                if (H5S_SELECT_ITER_GET_SEQ_LIST(mem_iter, H5FD_SEQ_LIST_LEN, SIZE_MAX, &mem_nseq, &seq_nelem,
                                                 mem_off, mem_len) < 0)
                    HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, FAIL, "sequence length generation failed");
                assert(mem_nseq > 0);

                mem_seq_i = 0;
            }
            assert(mem_seq_i < mem_nseq);

            
            io_len = MIN(file_len[file_seq_i], mem_len[mem_seq_i]);

            
            if (use_vector) {
                
                if (vec_arr_nused == vec_arr_nalloc) {
                    
                    if (addrs == addrs_local) {
                        assert(sizes == sizes_local);
                        assert(vec_bufs == vec_bufs_local);

                        
                        if (NULL == (addrs = H5MM_malloc(sizeof(addrs_local) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for address list");
                        if (NULL == (sizes = H5MM_malloc(sizeof(sizes_local) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for size list");
                        if (NULL == (vec_bufs = H5MM_malloc(sizeof(vec_bufs_local) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory allocation failed for buffer list");

                        
                        (void)H5MM_memcpy(addrs, addrs_local, sizeof(addrs_local));
                        (void)H5MM_memcpy(sizes, sizes_local, sizeof(sizes_local));
                        (void)H5MM_memcpy(vec_bufs, vec_bufs_local, sizeof(vec_bufs_local));
                    }
                    else {
                        void *tmp_ptr;

                        
                        if (NULL == (tmp_ptr = H5MM_realloc(addrs, vec_arr_nalloc * sizeof(*addrs) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory reallocation failed for address list");
                        addrs = tmp_ptr;
                        if (NULL == (tmp_ptr = H5MM_realloc(sizes, vec_arr_nalloc * sizeof(*sizes) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory reallocation failed for size list");
                        sizes = tmp_ptr;
                        if (NULL ==
                            (tmp_ptr = H5MM_realloc(vec_bufs, vec_arr_nalloc * sizeof(*vec_bufs) * 2)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL,
                                        "memory reallocation failed for buffer list");
                        vec_bufs = tmp_ptr;
                    }

                    
                    vec_arr_nalloc *= 2;
                }

                
                addrs[vec_arr_nused]    = offsets[i] + file_off[file_seq_i];
                sizes[vec_arr_nused]    = io_len;
                vec_bufs[vec_arr_nused] = (const void *)((const uint8_t *)buf + mem_off[mem_seq_i]);
                vec_arr_nused++;
            }
            else {
                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        
                        ret_value =
                            (file->cls->write)(file, type, dxpl_id, offsets[i] + file_off[file_seq_i], io_len,
                                               (const void *)((const uint8_t *)buf + mem_off[mem_seq_i]));
                    }
                H5_AFTER_USER_CB(FAIL)
                if (ret_value < 0)
                    HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "driver write request failed");
            }

            
            if (io_len == file_len[file_seq_i])
                file_seq_i++;
            else {
                file_off[file_seq_i] += io_len;
                file_len[file_seq_i] -= io_len;
            }

            
            if (io_len == mem_len[mem_seq_i])
                mem_seq_i++;
            else {
                mem_off[mem_seq_i] += io_len;
                mem_len[mem_seq_i] -= io_len;
            }
        }

        
        if (mem_seq_i < mem_nseq)
            HGOTO_ERROR(H5E_INTERNAL, H5E_BADVALUE, FAIL,
                        "file selection terminated before memory selection");

        
        if (H5S_SELECT_ITER_RELEASE(file_iter) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_CANTFREE, FAIL, "can't release file selection iterator");
        file_iter_init = false;
        if (H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_CANTFREE, FAIL, "can't release memory selection iterator");
        mem_iter_init = false;
    }

    
    if (use_vector) {
        uint32_t actual_selection_io_mode;

        H5_CHECK_OVERFLOW(vec_arr_nused, size_t, uint32_t);
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->write_vector)(file, dxpl_id, (uint32_t)vec_arr_nused, types, addrs,
                                                      sizes, vec_bufs);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "driver write vector request failed");

        
        if (type == H5FD_MEM_DRAW && count > 0) {
            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_VECTOR_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }
    else if (count > 0) {
        uint32_t no_selection_io_cause;
        uint32_t actual_selection_io_mode;

        
        H5CX_get_no_selection_io_cause(&no_selection_io_cause);
        no_selection_io_cause |= H5D_SEL_IO_NO_VECTOR_OR_SELECTION_IO_CB;
        H5CX_set_no_selection_io_cause(no_selection_io_cause);

        
        if (type == H5FD_MEM_DRAW) {
            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_SCALAR_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }

done:
    
    if (file_iter) {
        if (file_iter_init && H5S_SELECT_ITER_RELEASE(file_iter) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_CANTFREE, FAIL, "can't release file selection iterator");
        file_iter = H5FL_FREE(H5S_sel_iter_t, file_iter);
    }
    if (mem_iter) {
        if (mem_iter_init && H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
            HGOTO_ERROR(H5E_INTERNAL, H5E_CANTFREE, FAIL, "can't release memory selection iterator");
        mem_iter = H5FL_FREE(H5S_sel_iter_t, mem_iter);
    }

    
    if (use_vector) {
        if (addrs != addrs_local)
            addrs = H5MM_xfree(addrs);
        if (sizes != sizes_local)
            sizes = H5MM_xfree(sizes);
        if (vec_bufs != vec_bufs_local)
            vec_bufs = H5MM_xfree(vec_bufs);
    }

    
    assert(!addrs || addrs == addrs_local);
    assert(!sizes || sizes == sizes_local);
    assert(!vec_bufs || vec_bufs == vec_bufs_local);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_write_selection(H5FD_t *file, H5FD_mem_t type, uint32_t count, H5S_t **mem_spaces, H5S_t **file_spaces,
                     haddr_t offsets[], size_t element_sizes[], const void *bufs[])
{
    bool     offsets_cooked = false;
    hid_t    mem_space_ids_local[H5FD_LOCAL_SEL_ARR_LEN];
    hid_t   *mem_space_ids = mem_space_ids_local;
    hid_t    file_space_ids_local[H5FD_LOCAL_SEL_ARR_LEN];
    hid_t   *file_space_ids = file_space_ids_local;
    haddr_t  eoa;
    uint32_t num_spaces = 0;
    hid_t    dxpl_id    = H5I_INVALID_HID; 
    uint32_t i;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert((mem_spaces) || (count == 0));
    assert((file_spaces) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (element_sizes[0] != 0));
    assert((count == 0) || (bufs[0] != NULL));

    
    dxpl_id = H5CX_get_dxpl();

#ifndef H5_HAVE_PARALLEL
    
    if (0 == count)
        HGOTO_DONE(SUCCEED);
#endif 

    if (file->base_addr > 0) {
        
        for (i = 0; i < count; i++)
            offsets[i] += file->base_addr;
        offsets_cooked = true;
    }

    
    
    H5_BEFORE_USER_CB(FAIL)
        {
            eoa = (file->cls->get_eoa)(file, type);
        }
    H5_AFTER_USER_CB(FAIL)
    if (!H5_addr_defined(eoa))
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver get_eoa request failed");

    for (i = 0; i < count; i++)
        if ((offsets[i]) > eoa)
            HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL, "addr overflow, offsets[%d] = %llu, eoa = %llu", (int)i,
                        (unsigned long long)(offsets[i]), (unsigned long long)eoa);

    
    if (file->cls->write_selection) {
        uint32_t actual_selection_io_mode;

        
        if (count > sizeof(mem_space_ids_local) / sizeof(mem_space_ids_local[0])) {
            if (NULL == (mem_space_ids = H5MM_malloc(count * sizeof(hid_t))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed for dataspace list");
            if (NULL == (file_space_ids = H5MM_malloc(count * sizeof(hid_t))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed for dataspace list");
        }

        
        for (; num_spaces < count; num_spaces++) {
            if ((mem_space_ids[num_spaces] = H5I_register(H5I_DATASPACE, mem_spaces[num_spaces], true)) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register dataspace ID");

            if ((file_space_ids[num_spaces] = H5I_register(H5I_DATASPACE, file_spaces[num_spaces], true)) <
                0) {
                if (NULL == H5I_remove(mem_space_ids[num_spaces]))
                    HDONE_ERROR(H5E_VFL, H5E_CANTREMOVE, FAIL, "problem removing id");
                HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register dataspace ID");
            }
        }

        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->write_selection)(file, type, dxpl_id, count, mem_space_ids,
                                                         file_space_ids, offsets, element_sizes, bufs);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "driver write selection request failed");

        
        if (type == H5FD_MEM_DRAW) {
            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_SELECTION_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }
    else
        

        if (H5FD__write_selection_translate(SKIP_NO_CB, file, type, dxpl_id, count, mem_spaces, file_spaces,
                                            offsets, element_sizes, bufs) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "translation to vector or scalar write failed");

done:
    
    if (offsets_cooked) {
        assert(file->base_addr > 0);
        for (i = 0; i < count; i++)
            offsets[i] -= file->base_addr;
    }

    
    for (i = 0; i < num_spaces; i++) {
        if (NULL == H5I_remove(mem_space_ids[i]))
            HDONE_ERROR(H5E_VFL, H5E_CANTREMOVE, FAIL, "problem removing id");
        if (NULL == H5I_remove(file_space_ids[i]))
            HDONE_ERROR(H5E_VFL, H5E_CANTREMOVE, FAIL, "problem removing id");
    }
    if (mem_space_ids != mem_space_ids_local)
        mem_space_ids = H5MM_xfree(mem_space_ids);
    if (file_space_ids != file_space_ids_local)
        file_space_ids = H5MM_xfree(file_space_ids);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_write_selection_id(uint32_t skip_cb, H5FD_t *file, H5FD_mem_t type, uint32_t count,
                        hid_t mem_space_ids[], hid_t file_space_ids[], haddr_t offsets[],
                        size_t element_sizes[], const void *bufs[])
{
    bool     offsets_cooked = false;
    H5S_t   *mem_spaces_local[H5FD_LOCAL_SEL_ARR_LEN];
    H5S_t  **mem_spaces = mem_spaces_local;
    H5S_t   *file_spaces_local[H5FD_LOCAL_SEL_ARR_LEN];
    H5S_t  **file_spaces = file_spaces_local;
    haddr_t  eoa;
    hid_t    dxpl_id = H5I_INVALID_HID; 
    uint32_t i;
    uint32_t skip_selection_cb;
    uint32_t skip_vector_cb;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert((mem_space_ids) || (count == 0));
    assert((file_space_ids) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (element_sizes[0] != 0));
    assert((count == 0) || (bufs[0] != NULL));

    
    dxpl_id = H5CX_get_dxpl();

#ifndef H5_HAVE_PARALLEL
    
    if (0 == count)
        HGOTO_DONE(SUCCEED);
#endif 

    skip_selection_cb = skip_cb & SKIP_SELECTION_CB;
    skip_vector_cb    = skip_cb & SKIP_VECTOR_CB;

    if (file->base_addr > 0) {
        
        for (i = 0; i < count; i++)
            offsets[i] += file->base_addr;
        offsets_cooked = true;
    }

    
    
    H5_BEFORE_USER_CB(FAIL)
        {
            eoa = (file->cls->get_eoa)(file, type);
        }
    H5_AFTER_USER_CB(FAIL)
    if (!H5_addr_defined(eoa))
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver get_eoa request failed");

    for (i = 0; i < count; i++)
        if ((offsets[i]) > eoa)
            HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL, "addr overflow, offsets[%d] = %llu, eoa = %llu", (int)i,
                        (unsigned long long)(offsets[i]), (unsigned long long)eoa);

    
    if (!skip_selection_cb && file->cls->write_selection) {
        uint32_t actual_selection_io_mode;

        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (file->cls->write_selection)(file, type, dxpl_id, count, mem_space_ids,
                                                         file_space_ids, offsets, element_sizes, bufs);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "driver write selection request failed");

        
        if (type == H5FD_MEM_DRAW) {
            H5CX_get_actual_selection_io_mode(&actual_selection_io_mode);
            actual_selection_io_mode |= H5D_SELECTION_IO;
            H5CX_set_actual_selection_io_mode(actual_selection_io_mode);
        }
    }
    else {
        

        
        if (count > sizeof(mem_spaces_local) / sizeof(mem_spaces_local[0])) {
            if (NULL == (mem_spaces = H5MM_malloc(count * sizeof(H5S_t *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed for dataspace list");
            if (NULL == (file_spaces = H5MM_malloc(count * sizeof(H5S_t *))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed for dataspace list");
        }

        
        for (i = 0; i < count; i++) {
            if (NULL == (mem_spaces[i] = (H5S_t *)H5I_object_verify(mem_space_ids[i], H5I_DATASPACE)))
                HGOTO_ERROR(H5E_VFL, H5E_BADTYPE, H5I_INVALID_HID, "can't retrieve memory dataspace from ID");
            if (NULL == (file_spaces[i] = (H5S_t *)H5I_object_verify(file_space_ids[i], H5I_DATASPACE)))
                HGOTO_ERROR(H5E_VFL, H5E_BADTYPE, H5I_INVALID_HID, "can't retrieve file dataspace from ID");
        }

        
        if (H5FD__write_selection_translate(skip_vector_cb, file, type, dxpl_id, count, mem_spaces,
                                            file_spaces, offsets, element_sizes, bufs) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "translation to vector or scalar write failed");
    }

done:
    
    if (offsets_cooked) {
        assert(file->base_addr > 0);
        for (i = 0; i < count; i++)
            offsets[i] -= file->base_addr;
    }

    
    if (mem_spaces != mem_spaces_local)
        mem_spaces = H5MM_xfree(mem_spaces);
    if (file_spaces != file_spaces_local)
        file_spaces = H5MM_xfree(file_spaces);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_read_vector_from_selection(H5FD_t *file, H5FD_mem_t type, uint32_t count, hid_t mem_space_ids[],
                                hid_t file_space_ids[], haddr_t offsets[], size_t element_sizes[],
                                void *bufs[])
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert((mem_space_ids) || (count == 0));
    assert((file_space_ids) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (element_sizes[0] != 0));
    assert((count == 0) || (bufs[0] != NULL));

    
    
    if (H5FD_read_selection_id(SKIP_SELECTION_CB, file, type, count, mem_space_ids, file_space_ids, offsets,
                               element_sizes, bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "file selection read request failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)

} 

herr_t
H5FD_write_vector_from_selection(H5FD_t *file, H5FD_mem_t type, uint32_t count, hid_t mem_space_ids[],
                                 hid_t file_space_ids[], haddr_t offsets[], size_t element_sizes[],
                                 const void *bufs[])
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert((mem_space_ids) || (count == 0));
    assert((file_space_ids) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (element_sizes[0] != 0));
    assert((count == 0) || (bufs[0] != NULL));

    
    
    if (H5FD_write_selection_id(SKIP_SELECTION_CB, file, type, count, mem_space_ids, file_space_ids, offsets,
                                element_sizes, bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "file selection write request failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)

} 

herr_t
H5FD_read_from_selection(H5FD_t *file, H5FD_mem_t type, uint32_t count, hid_t mem_space_ids[],
                         hid_t file_space_ids[], haddr_t offsets[], size_t element_sizes[], void *bufs[])
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert((mem_space_ids) || (count == 0));
    assert((file_space_ids) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (element_sizes[0] != 0));
    assert((count == 0) || (bufs[0] != NULL));

    
    
    if (H5FD_read_selection_id(SKIP_SELECTION_CB | SKIP_VECTOR_CB, file, type, count, mem_space_ids,
                               file_space_ids, offsets, element_sizes, bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_READERROR, FAIL, "file selection read request failed");

done:

    FUNC_LEAVE_NOAPI(ret_value)

} 

herr_t
H5FD_write_from_selection(H5FD_t *file, H5FD_mem_t type, uint32_t count, hid_t mem_space_ids[],
                          hid_t file_space_ids[], haddr_t offsets[], size_t element_sizes[],
                          const void *bufs[])
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert((mem_space_ids) || (count == 0));
    assert((file_space_ids) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (element_sizes[0] != 0));
    assert((count == 0) || (bufs[0] != NULL));

    
    
    if (H5FD_write_selection_id(SKIP_SELECTION_CB | SKIP_VECTOR_CB, file, type, count, mem_space_ids,
                                file_space_ids, offsets, element_sizes, bufs) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_WRITEERROR, FAIL, "file selection write request failed");

done:

    FUNC_LEAVE_NOAPI(ret_value)

} 

herr_t
H5FD_set_eoa(H5FD_t *file, H5FD_mem_t type, haddr_t addr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(file && file->cls);
    assert(H5_addr_defined(addr) && addr <= file->maxaddr);

    
    H5_BEFORE_USER_CB(FAIL)
        {
            
            ret_value = (file->cls->set_eoa)(file, type, addr + file->base_addr);
        }
    H5_AFTER_USER_CB(FAIL)
    if (ret_value < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, FAIL, "driver set_eoa request failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5FD_get_eoa(const H5FD_t *file, H5FD_mem_t type)
{
    haddr_t ret_value = HADDR_UNDEF; 

    FUNC_ENTER_NOAPI(HADDR_UNDEF)

    assert(file && file->cls);

    
    H5_BEFORE_USER_CB(HADDR_UNDEF)
        {
            
            ret_value = (file->cls->get_eoa)(file, type);
        }
    H5_AFTER_USER_CB(HADDR_UNDEF)
    if (!H5_addr_defined(ret_value))
        HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, HADDR_UNDEF, "driver get_eoa request failed");

    
    ret_value -= file->base_addr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5FD_get_eof(const H5FD_t *file, H5FD_mem_t type)
{
    haddr_t ret_value = HADDR_UNDEF; 

    FUNC_ENTER_NOAPI(HADDR_UNDEF)

    assert(file && file->cls);

    
    if (file->cls->get_eof) {
        
        H5_BEFORE_USER_CB(HADDR_UNDEF)
            {
                ret_value = (file->cls->get_eof)(file, type);
            }
        H5_AFTER_USER_CB(HADDR_UNDEF)
        if (!H5_addr_defined(ret_value))
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, HADDR_UNDEF, "driver get_eof request failed");
    }
    else
        ret_value = file->maxaddr;

    
    ret_value -= file->base_addr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_driver_query(const H5FD_class_t *driver, unsigned long *flags )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(driver);
    assert(flags);

    
    if (driver->query) {
        
        H5_BEFORE_USER_CB_NOERR(FAIL)
            {
                ret_value = (driver->query)(NULL, flags);
            }
        H5_AFTER_USER_CB_NOERR(FAIL)
    }
    else
        *flags = 0;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5FD__srt_tmp_cmp(const void *element_1, const void *element_2)
{
    haddr_t addr_1    = ((const H5FD_srt_tmp_t *)element_1)->addr;
    haddr_t addr_2    = ((const H5FD_srt_tmp_t *)element_2)->addr;
    int     ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(H5_addr_defined(addr_1));
    assert(H5_addr_defined(addr_2));

    
    if (H5_addr_gt(addr_1, addr_2))
        ret_value = 1;
    else if (H5_addr_lt(addr_1, addr_2))
        ret_value = -1;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__sort_io_req_real(size_t count, haddr_t *addrs, bool *was_sorted, struct H5FD_srt_tmp_t **srt_tmp)
{
    size_t i;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    

    
    for (i = 1; i < count; i++) {
        assert(H5_addr_defined(addrs[i - 1]));

        if (H5_addr_gt(addrs[i - 1], addrs[i]))
            break;
        else if (H5_addr_eq(addrs[i - 1], addrs[i]))
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "duplicate addr in selections");
    }

    
    if (i >= count)
        *was_sorted = true;
    else
        *was_sorted = false;

    if (!(*was_sorted)) {
        size_t srt_tmp_size;

        srt_tmp_size = (count * sizeof(struct H5FD_srt_tmp_t));

        if (NULL == (*srt_tmp = (H5FD_srt_tmp_t *)malloc(srt_tmp_size)))

            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "can't alloc srt_tmp");

        for (i = 0; i < count; i++) {
            (*srt_tmp)[i].addr  = addrs[i];
            (*srt_tmp)[i].index = i;
        }

        
        qsort(*srt_tmp, count, sizeof(struct H5FD_srt_tmp_t), H5FD__srt_tmp_cmp);

        
        i = 1;

        for (i = 1; i < count; i++) {
            assert(H5_addr_lt((*srt_tmp)[i - 1].addr, (*srt_tmp)[i].addr));

            if (H5_addr_eq(addrs[i - 1], addrs[i]))
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "duplicate addrs in array");
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)

} 

herr_t
H5FD_sort_vector_io_req(bool *vector_was_sorted, uint32_t _count, H5FD_mem_t types[], haddr_t addrs[],
                        size_t sizes[], H5_flexible_const_ptr_t bufs[], H5FD_mem_t **s_types_ptr,
                        haddr_t **s_addrs_ptr, size_t **s_sizes_ptr, H5_flexible_const_ptr_t **s_bufs_ptr)
{
    herr_t                 ret_value = SUCCEED; 
    size_t                 count     = (size_t)_count;
    size_t                 i;
    struct H5FD_srt_tmp_t *srt_tmp = NULL;

    FUNC_ENTER_NOAPI(FAIL)

    

    assert(vector_was_sorted);

    assert((types) || (count == 0));
    assert((addrs) || (count == 0));
    assert((sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (sizes[0] != 0));
    assert((count == 0) || (types[0] != H5FD_MEM_NOLIST));

    assert((count == 0) || ((s_types_ptr) && (NULL == *s_types_ptr)));
    assert((count == 0) || ((s_addrs_ptr) && (NULL == *s_addrs_ptr)));
    assert((count == 0) || ((s_sizes_ptr) && (NULL == *s_sizes_ptr)));
    assert((count == 0) || ((s_bufs_ptr) && (NULL == *s_bufs_ptr)));

    
    if (H5FD__sort_io_req_real(count, addrs, vector_was_sorted, &srt_tmp) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sorting error in selection offsets");

    if (*vector_was_sorted) {

        *s_types_ptr = types;
        *s_addrs_ptr = addrs;
        *s_sizes_ptr = sizes;
        *s_bufs_ptr  = bufs;
    }
    else {

        
        size_t j;
        size_t fixed_size_index = count;
        size_t fixed_type_index = count;

        if ((NULL == (*s_types_ptr = (H5FD_mem_t *)malloc(count * sizeof(H5FD_mem_t)))) ||
            (NULL == (*s_addrs_ptr = (haddr_t *)malloc(count * sizeof(haddr_t)))) ||
            (NULL == (*s_sizes_ptr = (size_t *)malloc(count * sizeof(size_t)))) ||
            (NULL ==
             (*s_bufs_ptr = (H5_flexible_const_ptr_t *)malloc(count * sizeof(H5_flexible_const_ptr_t))))) {

            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "can't alloc sorted vector(s)");
        }

        assert(sizes[0] != 0);
        assert(types[0] != H5FD_MEM_NOLIST);

        
        for (i = 1; i < count && ((fixed_size_index == count) || (fixed_type_index == count)); i++) {
            if ((fixed_size_index == count) && (sizes[i] == 0))
                fixed_size_index = i - 1;
            if ((fixed_type_index == count) && (types[i] == H5FD_MEM_NOLIST))
                fixed_type_index = i - 1;
        }

        assert(fixed_size_index <= count);
        assert(fixed_type_index <= count);

        
        for (i = 0; i < count; i++) {

            j = srt_tmp[i].index;

            (*s_types_ptr)[i] = types[MIN(j, fixed_type_index)];
            (*s_addrs_ptr)[i] = addrs[j];
            (*s_sizes_ptr)[i] = sizes[MIN(j, fixed_size_index)];
            (*s_bufs_ptr)[i]  = bufs[j];
        }
    }

done:
    if (srt_tmp) {

        free(srt_tmp);
        srt_tmp = NULL;
    }

    
    if ((ret_value != SUCCEED) && (!(*vector_was_sorted))) {

        
        if (*s_types_ptr) {

            free(*s_types_ptr);
            *s_types_ptr = NULL;
        }

        if (*s_addrs_ptr) {

            free(*s_addrs_ptr);
            *s_addrs_ptr = NULL;
        }

        if (*s_sizes_ptr) {

            free(*s_sizes_ptr);
            *s_sizes_ptr = NULL;
        }

        if (*s_bufs_ptr) {

            free(*s_bufs_ptr);
            *s_bufs_ptr = NULL;
        }
    }

    FUNC_LEAVE_NOAPI(ret_value)

} 

herr_t
H5FD_sort_selection_io_req(bool *selection_was_sorted, size_t count, hid_t mem_space_ids[],
                           hid_t file_space_ids[], haddr_t offsets[], size_t element_sizes[],
                           H5_flexible_const_ptr_t bufs[], hid_t **s_mem_space_ids_ptr,
                           hid_t **s_file_space_ids_ptr, haddr_t **s_offsets_ptr,
                           size_t **s_element_sizes_ptr, H5_flexible_const_ptr_t **s_bufs_ptr)
{
    size_t                 i;
    struct H5FD_srt_tmp_t *srt_tmp   = NULL;
    herr_t                 ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    

    assert(selection_was_sorted);

    assert((mem_space_ids) || (count == 0));
    assert((file_space_ids) || (count == 0));
    assert((offsets) || (count == 0));
    assert((element_sizes) || (count == 0));
    assert((bufs) || (count == 0));

    
    assert((count == 0) || (element_sizes[0] != 0));
    assert((count == 0) || (bufs[0].cvp != NULL));

    assert((count == 0) || ((s_mem_space_ids_ptr) && (NULL == *s_mem_space_ids_ptr)));
    assert((count == 0) || ((s_file_space_ids_ptr) && (NULL == *s_file_space_ids_ptr)));
    assert((count == 0) || ((s_offsets_ptr) && (NULL == *s_offsets_ptr)));
    assert((count == 0) || ((s_element_sizes_ptr) && (NULL == *s_element_sizes_ptr)));
    assert((count == 0) || ((s_bufs_ptr) && (NULL == *s_bufs_ptr)));

    
    if (H5FD__sort_io_req_real(count, offsets, selection_was_sorted, &srt_tmp) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "sorting error in selection offsets");

    if (*selection_was_sorted) {

        *s_mem_space_ids_ptr  = mem_space_ids;
        *s_file_space_ids_ptr = file_space_ids;
        *s_offsets_ptr        = offsets;
        *s_element_sizes_ptr  = element_sizes;
        *s_bufs_ptr           = bufs;
    }
    else {

        
        size_t j;
        size_t fixed_element_sizes_index = count;
        size_t fixed_bufs_index          = count;

        if ((NULL == (*s_mem_space_ids_ptr = (hid_t *)malloc(count * sizeof(hid_t)))) ||
            (NULL == (*s_file_space_ids_ptr = (hid_t *)malloc(count * sizeof(hid_t)))) ||
            (NULL == (*s_offsets_ptr = (haddr_t *)malloc(count * sizeof(haddr_t)))) ||
            (NULL == (*s_element_sizes_ptr = (size_t *)malloc(count * sizeof(size_t)))) ||
            (NULL ==
             (*s_bufs_ptr = (H5_flexible_const_ptr_t *)malloc(count * sizeof(H5_flexible_const_ptr_t))))) {

            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "can't alloc sorted selection(s)");
        }

        assert(element_sizes[0] != 0);
        assert(bufs[0].cvp != NULL);

        
        for (i = 1; i < count && ((fixed_element_sizes_index == count) || (fixed_bufs_index == count)); i++) {
            if ((fixed_element_sizes_index == count) && (element_sizes[i] == 0))
                fixed_element_sizes_index = i - 1;
            if ((fixed_bufs_index == count) && (bufs[i].cvp == NULL))
                fixed_bufs_index = i - 1;
        }

        assert(fixed_element_sizes_index <= count);
        assert(fixed_bufs_index <= count);

        
        for (i = 0; i < count; i++) {

            j = srt_tmp[i].index;

            (*s_mem_space_ids_ptr)[i]  = mem_space_ids[j];
            (*s_file_space_ids_ptr)[i] = file_space_ids[j];
            (*s_offsets_ptr)[i]        = offsets[j];
            (*s_element_sizes_ptr)[i]  = element_sizes[MIN(j, fixed_element_sizes_index)];
            (*s_bufs_ptr)[i]           = bufs[MIN(j, fixed_bufs_index)];
        }
    }

done:
    if (srt_tmp) {
        free(srt_tmp);
        srt_tmp = NULL;
    }

    
    if ((ret_value != SUCCEED) && (!(*selection_was_sorted))) {

        
        if (*s_mem_space_ids_ptr) {
            free(*s_mem_space_ids_ptr);
            *s_mem_space_ids_ptr = NULL;
        }

        if (*s_file_space_ids_ptr) {
            free(*s_file_space_ids_ptr);
            *s_file_space_ids_ptr = NULL;
        }

        if (*s_offsets_ptr) {
            free(*s_offsets_ptr);
            *s_offsets_ptr = NULL;
        }

        if (*s_element_sizes_ptr) {
            free(*s_element_sizes_ptr);
            *s_element_sizes_ptr = NULL;
        }

        if (*s_bufs_ptr) {
            free(*s_bufs_ptr);
            *s_bufs_ptr = NULL;
        }
    }

    FUNC_LEAVE_NOAPI(ret_value)

} 

herr_t
H5FD_delete(const char *filename, hid_t fapl_id)
{
    H5FD_class_t      *driver;              
    H5FD_driver_prop_t driver_prop;         
    H5P_genplist_t    *plist;               
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    

    assert(filename);

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");

    
    if (H5P_peek(plist, H5F_ACS_FILE_DRV_NAME, &driver_prop) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get driver ID & info");

    
    if (NULL == (driver = (H5FD_class_t *)H5I_object(driver_prop.driver_id)))
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "invalid driver ID in file access property list");
    if (NULL == driver->del)
        HGOTO_ERROR(H5E_VFL, H5E_UNSUPPORTED, FAIL, "file driver has no 'del' method");

    
    H5_BEFORE_USER_CB(FAIL)
        {
            
            ret_value = (driver->del)(filename, fapl_id);
        }
    H5_AFTER_USER_CB(FAIL)
    if (ret_value < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTDELETEFILE, FAIL, "delete failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_check_plugin_load(const H5FD_class_t *cls, const H5PL_key_t *key, bool *success)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(cls);
    assert(key);
    assert(success);

    
    if (key->vfd.kind == H5FD_GET_DRIVER_BY_NAME) {
        
        if (cls->name && !strcmp(cls->name, key->vfd.u.name))
            *success = true;
    }
    else {
        
        assert(key->vfd.kind == H5FD_GET_DRIVER_BY_VALUE);

        
        if (cls->value == key->vfd.u.value)
            *success = true;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5FD__get_driver_cb(void *obj, hid_t id, void *_op_data)
{
    H5FD_get_driver_ud_t *op_data   = (H5FD_get_driver_ud_t *)_op_data; 
    H5FD_class_t         *cls       = (H5FD_class_t *)obj;
    int                   ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE_NOERR

    if (H5FD_GET_DRIVER_BY_NAME == op_data->key.kind) {
        if (0 == strcmp(cls->name, op_data->key.u.name)) {
            op_data->found_id = id;
            ret_value         = H5_ITER_STOP;
        } 
    }     
    else {
        assert(H5FD_GET_DRIVER_BY_VALUE == op_data->key.kind);
        if (cls->value == op_data->key.u.value) {
            op_data->found_id = id;
            ret_value         = H5_ITER_STOP;
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5FD_register_driver_by_name(const char *name, bool app_ref)
{
    htri_t driver_is_registered = false;
    hid_t  driver_id            = H5I_INVALID_HID;
    hid_t  ret_value            = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    
    if ((driver_is_registered = H5FD_is_driver_registered_by_name(name, &driver_id)) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_BADITER, H5I_INVALID_HID, "can't check if driver is already registered");

    
    if (driver_is_registered) {
        assert(driver_id >= 0);

        if (H5I_inc_ref(driver_id, app_ref) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINC, H5I_INVALID_HID, "unable to increment ref count on VFD");
    } 
    else {
        H5PL_key_t          key;
        const H5FD_class_t *cls;

        
        key.vfd.kind   = H5FD_GET_DRIVER_BY_NAME;
        key.vfd.u.name = name;
        if (NULL == (cls = (const H5FD_class_t *)H5PL_load(H5PL_TYPE_VFD, &key)))
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, H5I_INVALID_HID, "unable to load VFD");

        
        if ((driver_id = H5FD_register(cls, sizeof(*cls), app_ref)) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register VFD ID");
    } 

    ret_value = driver_id;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5FD_register_driver_by_value(H5FD_class_value_t value, bool app_ref)
{
    htri_t driver_is_registered = false;
    hid_t  driver_id            = H5I_INVALID_HID;
    hid_t  ret_value            = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    
    if ((driver_is_registered = H5FD_is_driver_registered_by_value(value, &driver_id)) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_BADITER, H5I_INVALID_HID, "can't check if driver is already registered");

    
    if (driver_is_registered) {
        assert(driver_id >= 0);

        if (H5I_inc_ref(driver_id, app_ref) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINC, H5I_INVALID_HID, "unable to increment ref count on VFD");
    } 
    else {
        H5PL_key_t          key;
        const H5FD_class_t *cls;

        
        key.vfd.kind    = H5FD_GET_DRIVER_BY_VALUE;
        key.vfd.u.value = value;
        if (NULL == (cls = (const H5FD_class_t *)H5PL_load(H5PL_TYPE_VFD, &key)))
            HGOTO_ERROR(H5E_VFL, H5E_CANTINIT, H5I_INVALID_HID, "unable to load VFD");

        
        if ((driver_id = H5FD_register(cls, sizeof(*cls), app_ref)) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register VFD ID");
    } 

    ret_value = driver_id;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5FD_is_driver_registered_by_name(const char *driver_name, hid_t *registered_id)
{
    H5FD_get_driver_ud_t op_data;           
    htri_t               ret_value = false; 

    FUNC_ENTER_NOAPI(FAIL)

    
    op_data.key.kind   = H5FD_GET_DRIVER_BY_NAME;
    op_data.key.u.name = driver_name;
    op_data.found_id   = H5I_INVALID_HID;

    
    if (H5I_iterate(H5I_VFL, H5FD__get_driver_cb, &op_data, false) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_BADITER, FAIL, "can't iterate over VFDs");

    
    if (op_data.found_id != H5I_INVALID_HID) {
        if (registered_id)
            *registered_id = op_data.found_id;
        ret_value = true;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5FD_is_driver_registered_by_value(H5FD_class_value_t driver_value, hid_t *registered_id)
{
    H5FD_get_driver_ud_t op_data;           
    htri_t               ret_value = false; 

    FUNC_ENTER_NOAPI(FAIL)

    
    op_data.key.kind    = H5FD_GET_DRIVER_BY_VALUE;
    op_data.key.u.value = driver_value;
    op_data.found_id    = H5I_INVALID_HID;

    
    if (H5I_iterate(H5I_VFL, H5FD__get_driver_cb, &op_data, false) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_BADITER, FAIL, "can't iterate over VFDs");

    
    if (op_data.found_id != H5I_INVALID_HID) {
        if (registered_id)
            *registered_id = op_data.found_id;
        ret_value = true;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5FD_get_driver_id_by_name(const char *name, bool is_api)
{
    H5FD_get_driver_ud_t op_data;
    hid_t                ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    
    op_data.key.kind   = H5FD_GET_DRIVER_BY_NAME;
    op_data.key.u.name = name;
    op_data.found_id   = H5I_INVALID_HID;

    
    if (H5I_iterate(H5I_VFL, H5FD__get_driver_cb, &op_data, false) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_BADITER, H5I_INVALID_HID, "can't iterate over VFL drivers");

    
    if (op_data.found_id != H5I_INVALID_HID) {
        ret_value = op_data.found_id;
        if (H5I_inc_ref(ret_value, is_api) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINC, H5I_INVALID_HID, "unable to increment ref count on VFL driver");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5FD_get_driver_id_by_value(H5FD_class_value_t value, bool is_api)
{
    H5FD_get_driver_ud_t op_data;
    hid_t                ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    
    op_data.key.kind    = H5FD_GET_DRIVER_BY_VALUE;
    op_data.key.u.value = value;
    op_data.found_id    = H5I_INVALID_HID;

    
    if (H5I_iterate(H5I_VFL, H5FD__get_driver_cb, &op_data, false) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_BADITER, H5I_INVALID_HID, "can't iterate over VFL drivers");

    
    if (op_data.found_id != H5I_INVALID_HID) {
        ret_value = op_data.found_id;
        if (H5I_inc_ref(ret_value, is_api) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTINC, H5I_INVALID_HID, "unable to increment ref count on VFL driver");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
