% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_ensemble_weights.R
\name{compute.ensemble.weights}
\alias{compute.ensemble.weights}
\title{Compute model averaging weights}
\usage{
compute.ensemble.weights(
  fit.list,
  type = c("bma", "pseudobma", "pseudobma+", "stacking"),
  prior.prob = NULL,
  bridge.args = NULL,
  loo.args = NULL,
  loo.wts.args = NULL,
  iter_warmup = 1000,
  iter_sampling = 1000,
  chains = 4,
  ...
)
}
\arguments{
\item{fit.list}{a list of model fit objects returned by functions in the \code{hdbayes} package. Each fit contains
posterior samples from a generalized linear model (GLM) (e.g., via \code{\link[=glm.pp]{glm.pp()}}), an accelerated
failure time (AFT) model (e.g., via \code{\link[=aft.pp]{aft.pp()}}), a piecewise exponential (PWE) model (e.g., via
\code{\link[=pwe.pp]{pwe.pp()}}), or a mixture cure rate model with a PWE component for the non-cured population (e.g.,
via \code{\link[=curepwe.pp]{curepwe.pp()}}). Each fit also includes two attributes: \code{data}, a list of variables specified
in the data block of the Stan program, and \code{model}, a character string indicating the model name.
To compute pseudo-BMA, pseudo-BMA+, or stacking weights, the fitting function must be called with
\code{get.loglik = TRUE}.}

\item{type}{a character string specifying the ensemble method used to compute model weights. Options are "bma"
(Bayesian model averaging (BMA)), "pseudobma" (pseudo-BMA without the Bayesian bootstrap), "pseudobma+"
(pseudo-BMA with the Bayesian bootstrap), and "stacking".}

\item{prior.prob}{a numeric vector of prior model probabilities, used only when \code{type = "bma"}. Must be non-negative
and sum to 1. If set to NULL, a uniform prior is used (i.e., all models are equally likely). Defaults
to NULL.}

\item{bridge.args}{a \code{list} of optional arguments (excluding \code{samples}, \code{log_posterior}, \code{data}, \code{lb}, and \code{ub}) to be
passed to \code{\link[bridgesampling:bridge_sampler]{bridgesampling::bridge_sampler()}}. These arguments are used when estimating the log marginal
likelihood, which is required if \code{type = "bma"}.}

\item{loo.args}{a \code{list} of optional arguments (excluding \code{x}) to be passed to \code{\link[loo:loo]{loo::loo()}} when computing pseudo-BMA,
pseudo-BMA+, or stacking weights.}

\item{loo.wts.args}{a \code{list} of optional arguments (excluding \code{x}, \code{method}, and \code{BB}) to be passed to \code{\link[loo:loo_model_weights]{loo::loo_model_weights()}}
when computing pseudo-BMA, pseudo-BMA+, or stacking weights.}

\item{iter_warmup}{number of warmup iterations to run per chain. Used only when computing the log marginal likelihood
(i.e., when \code{type = "bma"}). Defaults to 1000. See the argument \code{iter_warmup} in \code{sample()} method
in cmdstanr package.}

\item{iter_sampling}{number of post-warmup iterations to run per chain. Used only when computing the log marginal likelihood
(i.e., when \code{type = "bma"}). Defaults to 1000. See the argument \code{iter_sampling} in \code{sample()} method
in cmdstanr package.}

\item{chains}{number of Markov chains to run. Used only when computing the log marginal likelihood (i.e., when
\code{type = "bma"}). Defaults to 4. See the argument \code{chains} in \code{sample()} method in cmdstanr package.}

\item{...}{arguments passed to \code{sample()} method in cmdstanr package (e.g., \code{seed}, \code{refresh}, \code{init}). These are
used only when computing the log marginal likelihood (i.e., when \code{type = "bma"}).}
}
\value{
The function returns a \code{list} with the following objects

\describe{
\item{weights}{a numeric vector of normalized model weights corresponding to the models in \code{fit.list}. The names of the
weights are made unique based on the model identifiers.}

\item{type}{a character string indicating the method used to compute the model weights (e.g., "bma", "pseudobma", "pseudobma+",
or "stacking")}

\item{res.logml}{a list of log marginal likelihood estimation results, returned only when \code{type = "bma"}}

\item{loo.list}{a list of outputs from \code{\link[loo:loo]{loo::loo()}}, returned only when \code{type} is "pseudobma", "pseudobma+", or "stacking"}
}
}
\description{
Compute model averaging weights for a set of Bayesian models using Bayesian model averaging (BMA), pseudo-BMA,
pseudo-BMA+ (pseudo-BMA with the Bayesian bootstrap), or stacking. This function takes a list of model fit objects,
each containing posterior samples from a generalized linear model (GLM) or survival model, and returns normalized
weights that can be used for model comparison or combining posterior samples using functions like \code{\link[=sample.ensemble]{sample.ensemble()}}.
}
\details{
The input \code{fit.list} should be a list of outputs from model fitting functions in the \code{hdbayes} package, such as \code{\link[=glm.pp]{glm.pp()}}
(for generalized linear models), \code{\link[=aft.pp]{aft.pp()}} (for accelerated failure time models), \code{\link[=pwe.pp]{pwe.pp()}} (for piecewise exponential (PWE)
models), or \code{\link[=curepwe.pp]{curepwe.pp()}} (for mixture cure rate models with a PWE component for the non-cured population). To compute
pseudo-BMA, pseudo-BMA+, or stacking weights, each fit must include pointwise log-likelihood values. To ensure this, the
fitting function must be called with \code{get.loglik = TRUE}.

The arguments related to Markov chain Monte Carlo (MCMC) sampling are utilized to compute the logarithm of the normalizing
constant for BMA, if applicable.
}
\examples{
if (instantiate::stan_cmdstan_exists()) {
  if(requireNamespace("survival")){
    library(survival)
    data(E1684)
    data(E1690)
    ## replace 0 failure times with 0.50 days
    E1684$failtime[E1684$failtime == 0] = 0.50/365.25
    E1690$failtime[E1690$failtime == 0] = 0.50/365.25
    E1684$cage = as.numeric(scale(E1684$age))
    E1690$cage = as.numeric(scale(E1690$age))
    data_list = list(currdata = E1690, histdata = E1684)
    nbreaks = 3
    probs   = 1:nbreaks / nbreaks
    breaks  = as.numeric(
      quantile(E1690[E1690$failcens==1, ]$failtime, probs = probs)
    )
    breaks  = c(0, breaks)
    breaks[length(breaks)] = max(10000, 1000 * breaks[length(breaks)])
    fit.pwe.pp = pwe.pp(
      formula = survival::Surv(failtime, failcens) ~ treatment + sex + cage + node_bin,
      data.list = data_list,
      breaks = breaks,
      a0 = 0.5,
      get.loglik = TRUE,
      chains = 1, iter_warmup = 1000, iter_sampling = 2000
    )
    fit.pwe.post = pwe.post(
      formula = survival::Surv(failtime, failcens) ~ treatment + sex + cage + node_bin,
      data.list = data_list,
      breaks = breaks,
      get.loglik = TRUE,
      chains = 1, iter_warmup = 1000, iter_sampling = 2000
    )
    fit.aft.post = aft.post(
      formula = survival::Surv(failtime, failcens) ~ treatment + sex + cage + node_bin,
      data.list = data_list,
      dist = "weibull",
      beta.sd = 10,
      get.loglik = TRUE,
      chains = 1, iter_warmup = 1000, iter_sampling = 2000
    )
    compute.ensemble.weights(
      fit.list = list(fit.pwe.post, fit.pwe.pp, fit.aft.post),
      type = "pseudobma+",
      loo.args = list(save_psis = FALSE),
      loo.wts.args = list(optim_method="BFGS")
    )
  }
}
}
\references{
Yao, Y., Vehtari, A., Simpson, D., and Gelman, A. (2018). Using stacking to average Bayesian predictive distributions. Bayesian Analysis, 13(3), 917–1007.

Vehtari, A., Gelman, A., and Gabry, J. (2017). Practical Bayesian model evaluation using leave-one-out cross-validation and WAIC. Statistics and Computing, 27(5), 1413–1432.
}
\seealso{
\code{\link[=sample.ensemble]{sample.ensemble()}}
}
