#' @title Generate narrative description for an input using Gemini
#' @description Generate a narrative description for a given input (e.g., table, figure) by converting it to a suitable format and sending it to Gemini.
#' @param input Input object. For type = "table", a data.frame should be provided.
#' @param type Type of input. Default is "table". (e.g., "table", "figure")
#' @param prompt Optional prompt string to guide the narrative generation. If NULL, a default prompt is used.
#' @param ... Additional arguments passed to gemini()
#' @return Narrative description generated by Gemini
#' @export
#' @examples
#' \dontrun{
#' # Example data.frame
#' table_data <- data.frame(
#'   sort = c(1, 1, 2, 2, 2, 3, 3, 3, 4, 4),
#'   category = c(
#'     "Gender", "Gender", "Age Group", "Age Group", "Age Group",
#'     "Race", "Race", "Race", "Ethnicity", "Ethnicity"
#'   ),
#'   type = c(
#'     "F", "M", "<65", ">80", "65-80",
#'     "AMERICAN INDIAN OR ALASKA NATIVE", "BLACK OR AFRICAN AMERICAN", "WHITE",
#'     "HISPANIC OR LATINO", "NOT HISPANIC OR LATINO"
#'   ),
#'   Placebo = c(53, 33, 14, 30, 42, NA, 8, 78, 3, 83),
#'   Xanomeline_High_Dose = c(40, 44, 11, 18, 55, 1, 9, 74, 3, 81),
#'   Xanomeline_Low_Dose = c(50, 34, 8, 29, 47, NA, 6, 78, 6, 78),
#'   stringsAsFactors = FALSE # Do not convert strings to factors
#' )
#' gemini_narrative(table_data)
#' }
#' @importFrom knitr kable
#'
#' @seealso gemini
#'
gemini_narrative <- function(input, type = "table", prompt = NULL, ...) {
  if (type == "table") {
    if (!is.data.frame(input)) {
      stop("Input must be a data.frame when type is 'table'.")
    }
    # Generate markdown table string
    markdown_table_string <- paste0(knitr::kable(
      input,
      format = "pipe",
      col.names = colnames(input)
    ), collapse = "\n")
    # Use user prompt if provided, otherwise use default
    if (is.null(prompt)) {
      # Default prompt for table with a newline before the table
      prompt <- paste0(
        "Describe this table as if writing the Results section of a medical paper. Be concise, objective, data-focused, and structured by variable or group as needed.\n",
        markdown_table_string
      )
    } else {
      # If user provides prompt, append table markdown with a newline
      prompt <- paste0(prompt, "\n", markdown_table_string)
    }
    # Call gemini function
    result <- gemini(prompt, ...)
    return(result)
  } else if (type == "figure") {
    # Check if input is a valid image file path and a scalar string
    if (!is.character(input) || length(input) != 1 ||
      !file.exists(input) ||
      !(tools::file_ext(input) %in% c("png", "jpeg", "jpg", "webp", "heic", "heif"))) {
      stop("Input must be a single valid image file path (png, jpeg, jpg, webp, heic, heif) when type is 'figure'.")
    }
    # Determine image type
    ext <- tolower(tools::file_ext(input))
    if (ext == "jpg") ext <- "jpeg"
    # Use user prompt if provided, otherwise use default
    if (is.null(prompt)) {
      # Default prompt for figure
      prompt <- "Describe the figure in concise, objective Results-section style, focusing only on observable data like group differences, key stats, trends, and confidence intervals -no interpretation."
    }
    # Call gemini_image function (from gemini_image.R)
    result <- gemini_image(
      image = input,
      prompt = prompt,
      type = ext,
      ...
    )
    return(result)
  } else {
    stop("Currently only type = 'table' or 'figure' is supported.")
  }
}
