% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/behaviors.R
\name{create_edge}
\alias{create_edge}
\title{Configure Create Edge Behavior}
\usage{
create_edge(
  key = "create-edge",
  trigger = "drag",
  enable = FALSE,
  onCreate = NULL,
  onFinish = NULL,
  style = NULL,
  notify = FALSE,
  outputId = NULL,
  ...
)
}
\arguments{
\item{key}{Behavior unique identifier. Useful to modify this behavior from JS side.}

\item{trigger}{The way to trigger edge creation: "click" or "drag" (string, default: "drag").}

\item{enable}{Whether to enable this behavior (boolean or function, default: FALSE).
Our default implementation works in parallel with the \link{context_menu} plugin which is
responsible for activating the edge behavior when edge creation is selected.}

\item{onCreate}{Callback function for creating an edge, returns edge data (function, default: NULL).}

\item{onFinish}{Callback function for successfully creating an edge (function).
By default, we provide an internal implementation that disables the edge mode when the edge
creation is succesful so that it does not conflict with other drag behaviors.}

\item{style}{Style of the newly created edge (list, default: NULL).}

\item{notify}{Whether to show a feedback message in the ui.}

\item{outputId}{Manually pass the Shiny output ID. This is useful when the graph
is initialised outside the shiny render function and the ID cannot be automatically
inferred. This allows to set input values from the callback function
with the right namespace and graph ID. You must typically pass \code{session$ns("graphid")}
to ensure this also works in modules.}

\item{...}{Extra parameters. See \url{https://g6.antv.antgroup.com/en/manual/behavior/create-edge}.}
}
\value{
A list with the configuration settings for the create-edge behavior.
}
\description{
Creates a configuration object for the create-edge behavior in G6.
This allows users to create edges between nodes by clicking or dragging.
}
\note{
\link{create_edge}, \link{drag_element} and \link{drag_element_force} are incompatible by default,
as there triggers are the same. You can change the trigger to workaround this.
}
\examples{
# Basic configuration
config <- create_edge()
if (interactive()) {
  library(shiny)
  library(bslib)
  library(g6R)

  nodes <- list(
    list(
      id = "node1"
    ),
    list(
      id = "node2"
    )
  )

  modUI <- function(id) {
    ns <- NS(id)
    tagList(
      g6Output(ns("graph"))
    )
  }

  modServer <- function(id) {
    moduleServer(id, function(input, output, session) {
      output$graph <- renderG6({
        g6(nodes) |>
          g6_options(
            animation = FALSE,
            edge = edge_options(
              style = list(
                endArrow = TRUE
              )
            )
          ) |>
          g6_layout(d3_force_layout()) |>
          g6_behaviors(
            # avoid conflict with internal function
            g6R::create_edge(
              target = c("node", "combo", "canvas"),
              enable = JS(
                "(e) => {
                  return e.shiftKey;
                }"
              ),
              onFinish = JS(
                sprintf(
                  "(edge) => {
                    const graph = HTMLWidgets.find('#\%s').getWidget();
                    const targetType = graph.getElementType(edge.target);
                    // Avoid to create edges in combos. If so, we remove it
                    if (targetType === 'combo') {
                      graph.removeEdgeData([edge.id]);
                      return;
                    }
                    Shiny.setInputValue('\%s', edge);
                  }",
                  session$ns("graph"),
                  session$ns("added_edge")
                )
              )
            )
          )
      })

      observeEvent(input[["added_edge"]], {
        showNotification(
          sprintf("Edge dropped on: \%s", input[["added_edge"]]$targetType),
          type = "message"
        )
      })
    })
  }

  ui <- page_fluid(
    modUI("test")
  )

  server <- function(input, output, session) {
    modServer("test")
  }

  shinyApp(ui, server)
}
}
