#' @title as_fc
#' @description This function allows to initialize a flow chart given any database. It will create a fc object showing the number of rows of the database. If a database is not available, the user can instead directly enter the number of rows in the study.
#'
#' @param .data Data frame to be initialised as a flowchart.
#' @param N Number of rows of the study in case `.data` is `NULL`.
#' @param label Character or expression with the text that will be shown in the box.
#' @param text_pattern Character or expression defining the structure that will have the text in each of the boxes. It recognizes `label`, `n`, `N` and `perc` within brackets. For default it is `"{label}\n {n}"`. If `text_pattern` or `label` is an expression, the `label` is always placed at the beginning of the pattern, followed by a line break where the structure specified by `text_pattern` is placed.
#' @param just Justification for the text: left, center or right. Default is center.
#' @param text_color Color of the text. It is `"black"` by default. See the `col` parameter for [gpar].
#' @param text_fs Font size of the text. It is 8 by default. See the `fontsize` parameter for [gpar].
#' @param text_fface Font face of the text. It is 1 by default. See the `fontface` parameter for [gpar].
#' @param text_ffamily Changes the font family of the text. Default is `NA`. See the `fontfamily` parameter for [gpar].
#' @param text_padding Changes the text padding inside the box. Default is 1. This number has to be greater than 0.
#' @param bg_fill Box background color. It is white by default. See the `fill` parameter for [gpar].
#' @param border_color Box border color. It is `"black"` by default. See the `col` parameter for [gpar].
#' @param width Width of the box. If `NA`, it automatically adjusts to the content (default). Must be an object of class [unit] or a number between 0 and 1.
#' @param height Height of the box. If `NA`, it automatically adjusts to the content (default). Must be an object of class [unit] or a number between 0 and 1.
#' @param hide Logical value to hide the initial box or not. Default is FALSE. hide = TRUE can only be combined with [fc_split()].
#' @param title Add a title box to the initial box. Default is `NULL`.
#' @param x_title x-coordinate of the title box. Default is `0.1` (placed in the left).
#' @param text_color_title Color of the title text. It is `"black"` by default.
#' @param text_fs_title Font size of the title text. It is 8 by default.
#' @param text_fface_title Font face of the title text. It is 1 by default. See the `fontface` parameter for [gpar].
#' @param text_ffamily_title Changes the font family of the title text. Default is `NA`. See the `fontfamily` parameter for [gpar].
#' @param text_padding_title Changes the title text padding inside the box. Default is 1. This number has to be greater than 0.
#' @param bg_fill_title Title box background color. It is `"white"` by default.
#' @param border_color_title Title box border color. It is `"black"` by default.
#' @param width_title Width of the title box. If `NA`, it automatically adjusts to the content (default). Must be an object of class [unit] or a number between 0 and 1.
#' @param height_title Height of the title box. If `NA`, it automatically adjusts to the content (default). Must be an object of class [unit] or a number between 0 and 1.
#'
#' @return List with the dataset and the initialized flowchart parameters.
#'
#' @examples
#' safo |>
#' as_fc(label = "Patients assessed for eligibility") |>
#' fc_draw()
#'
#' @export

as_fc <- function(.data = NULL, N = NULL, label = "Initial dataframe", text_pattern = "{label}\n{N}", just = "center", text_color = "black", text_fs = 8, text_fface = 1, text_ffamily = NA, text_padding = 1, bg_fill = "white", border_color = "black", width = NA, height = NA, hide = FALSE, title = NULL, x_title = 0.1, text_color_title = "black", text_fs_title = 10, text_fface_title = 1, text_ffamily_title = NA, text_padding_title = 0.6, bg_fill_title = "#B3D1FF", border_color_title = "black", width_title = NA, height_title = NA) {

  if(is.null(.data) & is.null(N)) {
    cli::cli_abort("Either {.arg .data} or {.arg N} arguments must be specified.")
  }else if(!is.null(.data) & !is.null(N)) {
    cli::cli_abort("The {.arg .data} and {.arg N} arguments cannot be specified simultaneously.")
  }

  if(!is.null(.data)) {
    N <- nrow(.data)
  } else {
    .data <- tibble::tibble(id = 1:N)
  }

  if(text_padding == 0) {
    cli::cli_abort("Text padding cannot be equal to zero.")
  }

  if(!hide) {

    new_fc <- tibble::tibble(
      id = 1,
      x = 0.5,
      y = 0.5,
      n = N,
      N = N,
      perc = "100",
      type = "init",
      group = NA,
      just = just,
      text_color = text_color,
      text_fs = text_fs,
      text_fface = text_fface,
      text_ffamily = text_ffamily,
      text_padding = text_padding,
      bg_fill = bg_fill,
      border_color = border_color,
      width = width,
      height = height,
      end = TRUE
    )

    if(is.expression(label) | is.expression(text_pattern)) {

      if(is.expression(text_pattern)) {

        text_pattern_exp <- gsub("\\{label\\}", "", as.character(text_pattern)) |>
          stringr::str_glue(.envir = rlang::as_environment(new_fc))

        text_pattern_exp <- tryCatch(
          parse(text = text_pattern_exp),
          error = \(e) {
            list(as.character(text_pattern_exp))
          })

        new_fc <- new_fc |>
          dplyr::mutate(text = list(substitute(atop(x, y), list(x = label[[1]], y = text_pattern_exp[[1]]))))

      } else {

        text_pattern_exp <- gsub("\\{label\\}", "", text_pattern)

        new_fc <- new_fc |>
          dplyr::mutate(text = list(substitute(atop(x, y), list(x = label[[1]], y = stringr::str_glue(text_pattern_exp)))))

      }

    } else if(is.character(label) & is.character(text_pattern)) {

      new_fc <- new_fc |>
        dplyr::mutate(text = as.character(stringr::str_glue(text_pattern)))

    } else {

      cli::cli_abort("The {.arg label} and {.arg text_pattern} must be either characters or expressions.")

    }

    new_fc <- if(is.expression(label)) new_fc |> dplyr::mutate(label = list(label)) else new_fc |> dplyr::mutate(label = label)

    new_fc <- if(is.expression(text_pattern)) new_fc |> dplyr::mutate(text_pattern = list(text_pattern)) else new_fc |> dplyr::mutate(text_pattern = text_pattern)

    new_fc <- new_fc |>
      dplyr::relocate(c("label", "text_pattern", "text"), .after = "perc")

  } else {

    cli::cli_warn("{.code hide = TRUE} can only be combined with {.fn fc_split}")
    new_fc <- NULL

  }

  #If we have to add a title
  if(!is.null(title)) {

    new_fc2 <- new_fc |>
      dplyr::mutate(
        x = x_title,
        n = NA,
        N = NA,
        n = NA,
        N = NA,
        perc = NA,
        label = NA,
        text_pattern = NA,
        text = title,
        type = "title_init",
        group = NA,
        just = "center",
        text_color = text_color_title,
        text_fs = text_fs_title,
        text_fface = text_fface_title,
        text_ffamily = text_ffamily_title,
        text_padding = text_padding_title,
        bg_fill = bg_fill_title,
        border_color = border_color_title,
        width = width_title,
        height = height_title,
        end = FALSE
      )

    new_fc <- rbind(
      new_fc,
      new_fc2
    )

  }

  #Initialize flowchart as x is a dataframe
  object <- list(
    data = .data |>
      dplyr::ungroup(),
    fc = new_fc
  )

  class(object) <- c("fc")

  object

}
