#' Plot Fitted Probability Distributions for NS-FFA
#'
#' Generates a plot showing probability densities of a nonstationary model for selected 
#' time slices (left panel) and the data (right panel).
#'
#' @param results A fitted flood frequency model generated by [fit_lmoments()],
#' [fit_mle()] or [fit_gmle()].
#'
#' @param slices Years at which to plot the nonstationary probability model.
#'
#' @param show_line If `TRUE` (default), draw a fitted line through the data.
#'
#' @param ... Optional named arguments: 'title', 'xlabel', and 'ylabel'.
#'
#' @return `ggplot`; a plot showing:
#' - The likelihood function of the distribution plotted vertically on the left panel.
#' - The data, connected with a line if `show_line == TRUE`, on the right panel.
#'
#' @examples
#' data <- rnorm(n = 100, mean = 100, sd = 10) + seq(1, 100)
#' years <- seq(from = 1901, to = 2000)
#' ns_structure <- list(location = TRUE, scale = FALSE)
#'
#' results <- fit_mle(
#' 	   data, 
#' 	   "GEV", 
#' 	   ns_years = years, 
#' 	   ns_structure = ns_structure
#' )
#'
#' plot_nsffa_fit(results)
#'
#' @import ggplot2
#' @export
plot_nsffa_fit <- function(
	results,
	slices = c(1925, 1950, 1975, 2000),
	show_line = TRUE,
	...
) {

	# Define y-axis range
	y_range <- c(0, 1.1 * max(results$data))
	quantiles <- seq(0, 1.1 * max(results$data), length.out = 200)

	# Create a dataframe for the distribution
	df_list <- lapply(slices, function (slice) {

		density <- sapply(quantiles, function(x) {
			exp(utils_log_likelihood(
				x,
				results$distribution,
				results$params,
				slice,
				results$ns_structure
			))
		})

		data.frame(
			x = quantiles,
			y = density,
			slice = as.character(slice)
		)

	})

    distribution_df <- do.call(rbind, df_list)

	# Create a dataframe for the data
	data_df <- data.frame(x = results$ns_years, y = results$data)

	# Create a dataframe for the slices
	vlines_df <- data.frame(x = slices, slice = factor(as.character(slices)))

	# Capture optional arguments
	args <- list(...)

	# Set labels and title 
	title <- args$title %||% "Fitted Flood Frequency Model (NS-FFA)"
	xlabel <- args$xlabel %||% "Years"
	ylabel <- args$ylabel %||% expression(Streamflow ~ m^3/s)

	# Plot of distribution
    palette <- c("#541352", "#3A5E8C", "#2F9AA0", "#10A53D", "#FFCF20")

	p1 <- ggplot(distribution_df, aes(x = .data$x, y = .data$y)) +
		geom_line(
			aes(color = .data$slice),
			linewidth = 1,
			show.legend = FALSE
		) +
		scale_color_manual(values = palette[seq_along(slices)]) +
		labs(x = ylabel, y = "Density") +
		coord_flip()

	p1 <- add_theme(p1) + theme(
		axis.text.x = element_blank(),
		axis.ticks.x = element_blank(),
	)

	# Plot of data
	p2 <- ggplot(data_df, aes(x = .data$x, y = .data$y)) +
		geom_point(color = "dimgray", size = 2.25) + 
		{ if (show_line) geom_line(color = "dimgray", linewidth = 1) else NULL } +
		geom_vline(
			data = vlines_df,
			aes(xintercept = .data$x, color = .data$slice),
			linewidth = 1.2,
		) +
		scale_x_continuous(breaks = slices) +
		scale_y_continuous(limits = y_range) +
		scale_color_manual(values = palette[seq_along(slices)]) +
		labs(x = xlabel, y = "", title = title, color = "Year")

	p2 <- add_theme(p2) + theme(
		axis.text.y = element_blank(),
		axis.ticks.y = element_blank()
	)

	# Return the plot
	(p1 + p2) + 
	plot_layout(widths = c(1, 6)) + 
	plot_annotation(title = "")

} 

