% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cut_contMap_for_focal_time.R
\name{cut_contMap_for_focal_time}
\alias{cut_contMap_for_focal_time}
\title{Cut the phylogeny and continuous trait mapping for a given focal time in the past}
\usage{
cut_contMap_for_focal_time(contMap, focal_time, keep_tip_labels = TRUE)
}
\arguments{
\item{contMap}{Object of class \code{"contMap"}, typically generated with \code{\link[phytools:contMap]{phytools::contMap()}},
that contains a phylogenetic tree and associated continuous trait mapping.
The phylogenetic tree must be rooted and fully resolved/dichotomous,
but it does not need to be ultrametric (it can includes fossils).}

\item{focal_time}{Numerical. The time, in terms of time distance from the present,
for which the tree and mapping must be cut. It must be smaller than the root age of the phylogeny.}

\item{keep_tip_labels}{Logical. Specify whether terminal branches with a single descendant tip must retained their initial \code{tip.label}. Default is \code{TRUE}.}
}
\value{
The function returns the cut contMap as an object of class \code{"contMap"}.
It contains a \verb{$tree} element of classes \code{"simmap"} and \code{"phylo"}. This function updates and adds multiple useful sub-elements to the \verb{$tree} element.
\itemize{
\item \verb{$maps} An updated list of named numerical vectors. Provides the mapping of trait values along each remaining edge.
\item \verb{$mapped.edge} An updated matrix. Provides the evolutionary time spent across trait values (columns) along the remaining edges (rows).
\item \verb{$root_age} Integer. Stores the age of the root of the tree.
\item \verb{$nodes_ID_df} Data.frame with two columns. Provides the conversion from the \code{new_node_ID} to the \code{initial_node_ID}. Each row is a node.
\item \verb{$initial_nodes_ID} Vector of character strings. Provides the initial ID of internal nodes. Used to plot internal node IDs as labels with \code{\link[ape:nodelabels]{ape::nodelabels()}}.
\item \verb{$edges_ID_df} Data.frame with two columns. Provides the conversion from the \code{new_edge_ID} to the \code{initial_edge_ID}. Each row is an edge/branch.
\item \verb{$initial_edges_ID} Vector of character strings. Provides the initial ID of edges/branches. Used to plot edge/branch IDs as labels with \code{\link[ape:nodelabels]{ape::edgelabels()}}.
}
}
\description{
Cuts off all the branches of the phylogeny which are
younger than a specific time in the past (i.e. the \code{focal_time}).
Branches overlapping the \code{focal_time} are shorten to the \code{focal_time}.
Likewise, remove continuous trait mapping for the cut off branches
by updating the \verb{$tree$maps} and \verb{$tree$mapped.edge} elements.
}
\details{
The phylogenetic tree is cut for a specific time in the past (i.e. the \code{focal_time}).

When a branch with a single descendant tip is cut and \code{keep_tip_labels = TRUE},
the leaf left is labeled with the tip.label of the unique descendant tip.

When a branch with a single descendant tip is cut and \code{keep_tip_labels = FALSE},
the leaf left is labeled with the node ID of the unique descendant tip.

In all cases, when a branch with multiple descendant tips (i.e., a clade) is cut,
the leaf left is labeled with the node ID of the MRCA of the cut-off clade.

The continuous trait mapping is updated accordingly by removing mapping associated with the cut off branches.
}
\examples{
# ----- Prepare data ----- #

# Load mammals phylogeny and data from the R package motmot (data included in deepSTRAPP)
# Initial data source: Slater, 2013; DOI: 10.1111/2041-210X.12084

data(mammals, package = "deepSTRAPP")

mammals_tree <- mammals$mammal.phy
mammals_data <- setNames(object = mammals$mammal.mass$mean,
                         nm = row.names(mammals$mammal.mass))[mammals_tree$tip.label]

# Run a stochastic mapping based on a Brownian Motion model
# for Ancestral Trait Estimates to obtain a "contMap" object
mammals_contMap <- phytools::contMap(mammals_tree, x = mammals_data,
                                     res = 100, # Number of time steps
                                     plot = FALSE)

# Set focal time
focal_time <- 80

# ----- Example 1: keep_tip_labels = TRUE ----- #

# Cut contMap to 80 Mya while keeping tip.label
# on terminal branches with a unique descending tip.
updated_contMap <- cut_contMap_for_focal_time(contMap = mammals_contMap,
                                              focal_time = focal_time,
                                              keep_tip_labels = TRUE)

# Plot node labels on initial stochastic map with cut-off
plot_contMap(mammals_contMap, lwd = 2, fsize = c(0.5, 1))
ape::nodelabels(cex = 0.5)
abline(v = max(phytools::nodeHeights(mammals_contMap$tree)[,2]) - focal_time,
       col = "red", lty = 2, lwd = 2)

# Plot initial node labels on cut stochastic map
plot_contMap(updated_contMap, fsize = c(0.8, 1))
ape::nodelabels(cex = 0.8, text = updated_contMap$tree$initial_nodes_ID)

# ----- Example 2: keep_tip_labels = FALSE ----- #

# Cut contMap to 80 Mya while NOT keeping tip.label.
updated_contMap <- cut_contMap_for_focal_time(contMap = mammals_contMap,
                                              focal_time = focal_time,
                                              keep_tip_labels = FALSE)

# Plot node labels on initial stochastic map with cut-off
plot_contMap(mammals_contMap, fsize = c(0.5, 1))
ape::nodelabels(cex = 0.5)
abline(v = max(phytools::nodeHeights(mammals_contMap$tree)[,2]) - focal_time,
       col = "red", lty = 2, lwd = 2)

# Plot initial node labels on cut stochastic map
plot_contMap(updated_contMap, fsize = c(0.8, 1))
ape::nodelabels(cex = 0.8, text = updated_contMap$tree$initial_nodes_ID)

}
\seealso{
\code{\link[=cut_phylo_for_focal_time]{cut_phylo_for_focal_time()}} \code{\link[=extract_most_likely_trait_values_for_focal_time]{extract_most_likely_trait_values_for_focal_time()}}
\code{\link[=extract_most_likely_trait_values_from_contMap_for_focal_time]{extract_most_likely_trait_values_from_contMap_for_focal_time()}}

For a guided tutorial, see this vignette: \code{vignette("cut_phylogenies", package = "deepSTRAPP")}
}
\author{
Maël Doré
}
