% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_heatmap.R
\name{plot_heatmap}
\alias{plot_heatmap}
\title{Visualize a Distance or Similarity Matrix as a Heatmap with Clustering}
\usage{
plot_heatmap(
  dist_mat,
  max_n = 50,
  group = NULL,
  stratified_sampling = FALSE,
  main_title = NULL,
  palette = "YlOrRd",
  clustering_method = "complete",
  cluster_rows = TRUE,
  cluster_cols = TRUE,
  fontsize_row = 10,
  fontsize_col = 10,
  show_rownames = TRUE,
  show_colnames = TRUE,
  border_color = "grey60",
  annotation_legend = TRUE,
  seed = 123
)
}
\arguments{
\item{dist_mat}{A square distance matrix (numeric matrix) or a \code{dist} object.}

\item{max_n}{Integer. Maximum number of observations (rows/columns) to display.
If the matrix exceeds this size, a subset of \code{max_n} observations is selected.}

\item{group}{Optional vector or factor providing group labels for rows/columns, used for color annotation.}

\item{stratified_sampling}{Logical. If \code{TRUE} and \code{group} is provided, sampling is stratified by group.
Each group will contribute at least one observation if possible. Default is \code{FALSE}.}

\item{main_title}{Optional character string specifying the main title of the heatmap.}

\item{palette}{Character string specifying the RColorBrewer palette for heatmap cells. Default is \code{"YlOrRd"}.}

\item{clustering_method}{Character string specifying the hierarchical clustering method,
as accepted by \code{\link[stats]{hclust}} (e.g., \code{"complete"}, \code{"average"}, \code{"ward.D2"}).}

\item{cluster_rows}{Logical, whether to perform hierarchical clustering on rows. Default is \code{TRUE}.}

\item{cluster_cols}{Logical, whether to perform hierarchical clustering on columns. Default is \code{TRUE}.}

\item{fontsize_row}{Integer specifying the font size of row labels. Default is 10.}

\item{fontsize_col}{Integer specifying the font size of column labels. Default is 10.}

\item{show_rownames}{Logical, whether to display row names. Default is \code{TRUE}.}

\item{show_colnames}{Logical, whether to display column names. Default is \code{TRUE}.}

\item{border_color}{Color of the cell borders in the heatmap. Default is \code{"grey60"}.}

\item{annotation_legend}{Logical, whether to display the legend for group annotations. Default is \code{TRUE}.}

\item{seed}{Integer. Random seed used when sampling rows/columns if \code{max_n} is smaller than total observations. Default is 123.}
}
\value{
Invisibly returns the \code{pheatmap} object, allowing further customization if assigned.
}
\description{
This function creates a heatmap from a square distance or similarity matrix.
If a similarity matrix is provided, it should first be converted to a distance matrix by the user.
The function supports hierarchical clustering, group annotations, row/column sampling (random or stratified),
and various customization options.
}
\details{
The function works as follows:
\itemize{
  \item Converts \code{dist} objects to matrices automatically.
  \item Samples rows/columns if the matrix is larger than \code{max_n}. Sampling can be random or stratified by group.
  \item In stratified sampling mode, each group contributes at least one observation if possible.
  \item Supports row annotations for groups and automatically assigns colors.
  \item Uses \code{pheatmap} for plotting with customizable clustering, labels, fonts, and colors.
}

This function is used internally by \code{\link{visualize_distances}()} but can be called directly for advanced usage.
}
\examples{
# Example: Euclidean distance heatmap on iris
eucli_dist <- stats::dist(iris[, 1:4])
dbrobust::plot_heatmap(
  dist_mat = eucli_dist,
  max_n = 10,
  group = iris$Species,
  stratified_sampling = TRUE,
  main_title = "Euclidean Distance Heatmap",
  palette = "YlOrRd",
  clustering_method = "complete"
)

# Example: GGower distances with small subset
data("Data_HC_contamination", package = "dbrobust")
Data_small <- Data_HC_contamination[1:50, ]
cont_vars <- c("V1", "V2", "V3", "V4")
cat_vars  <- c("V5", "V6", "V7")
bin_vars  <- c("V8", "V9")
w <- Data_small$w_loop
dist_sq_ggower <- dbrobust::robust_distances(
  data = Data_small,
  cont_vars = cont_vars,
  bin_vars  = bin_vars,
  cat_vars  = cat_vars,
  w = w,
  alpha = 0.10,
  method = "ggower"
)
group_vec <- rep("Normal", nrow(dist_sq_ggower))
group_vec[attr(dist_sq_ggower, "outlier_idx")] <- "Outlier"
group_factor <- factor(group_vec, levels = c("Normal", "Outlier"))
dbrobust::plot_heatmap(
  dist_mat = sqrt(dist_sq_ggower),
  max_n = 20,
  group = group_factor,
  main_title = "GGower Heatmap with Outliers",
  palette = "YlOrRd",
  clustering_method = "complete",
  annotation_legend = TRUE,
  stratified_sampling = TRUE,
  seed = 123
)

}
\seealso{
\code{\link[stats]{hclust}} for hierarchical clustering methods.
\code{\link[pheatmap]{pheatmap}} for additional heatmap customization options.
\code{\link[RColorBrewer]{brewer.pal}} for available color palettes.
}
\keyword{internal}
