% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/make_euclidean.R
\name{make_euclidean}
\alias{make_euclidean}
\title{Force a Pairwise Squared Distance Matrix to Euclidean Form}
\usage{
make_euclidean(D, w, tol = 1e-10)
}
\arguments{
\item{D}{Numeric square matrix (n x n) of pairwise squared distances.
Must be symmetric with zeros on the diagonal.}

\item{w}{Numeric vector of weights (length n). Internally normalized to sum to 1.}

\item{tol}{Numeric tolerance for detecting negative eigenvalues (default: 1e-10).}
}
\value{
A list with components:
\describe{
  \item{D_euc}{Corrected pairwise squared Euclidean distance matrix (n x n).}
  \item{eigvals_before}{Eigenvalues of the weighted Gram matrix before correction.}
  \item{eigvals_after}{Eigenvalues of the weighted Gram matrix after correction.}
  \item{transformed}{Logical, TRUE if correction was applied, FALSE otherwise.}
}
}
\description{
Given a pairwise squared distance matrix \eqn{D} (where \eqn{D[i,j] = d(i,j)^2}),
this function ensures that \eqn{D} corresponds to a valid Euclidean squared
distance matrix. The correction is based on the weighted Gram matrix
\eqn{G_w = -\frac{1}{2} J_w D J_w^\top}, where \eqn{J_w = I_n - \mathbf{1} w^\top}
is the centering matrix defined by the weight vector \eqn{w}.
}
\details{
If the smallest eigenvalue \eqn{\lambda_{\min}} of \eqn{G_w} is below the
negative tolerance \code{-tol}, the function corrects \eqn{D} by adding a
constant shift to guarantee positive semi-definiteness of the Gram matrix,
following the approach of \insertCite{lingoes1971some}{dbrobust} and
\insertCite{mardia1978some}{dbrobust}:
\deqn{
D_{\text{new}} = D + 2 c \mathbf{1} \mathbf{1}^\top - 2 c I_n,
}
where \eqn{c = |\lambda_{\min}|}.
}
\examples{
# Load example dataset
data("Data_HC_contamination")

# Reduce dataset to first 50 rows
Data_small <- Data_HC_contamination[1:50, ]

# Select only continuous variables
cont_vars <- names(Data_small)[1:4]
Data_cont <- Data_small[, cont_vars]

# Compute squared Euclidean distance matrix
dist_mat <- as.matrix(dist(Data_cont))^2

# Introduce a small non-Euclidean distortion
dist_mat[1, 2] <- dist_mat[1, 2] * 0.5
dist_mat[2, 1] <- dist_mat[1, 2]

# Uniform weights
weights <- rep(1, nrow(Data_cont))

# Apply Euclidean correction
res <- make_euclidean(dist_mat, weights)

# Check results (minimum eigenvalues before/after)
res$transformed
min(res$eigvals_before)
min(res$eigvals_after)

# First 5x5 block of corrected matrix
round(res$D_euc[1:5, 1:5], 4)

}
\references{
\insertRef{lingoes1971some}{dbrobust}
\insertRef{mardia1978some}{dbrobust}
}
\seealso{
\code{\link[stats]{dist}}, \code{\link[base]{eigen}}, \code{\link{cmdscale}}
}
