# Posterior summaries for AppRaise

#' Posterior probability of significance
#'
#' Computes the posterior probability that the target parameter
#' exceeds the threshold for significance.
#'
#' @param mid_samples Numeric vector of indicator draws (0/1)
#'
#' @return Numeric probability
#'
#' @seealso
#' \itemize{
#'   \item \code{\link{posterior_mixture}}
#'   \item \code{vignette("appraise-introduction")}
#' }
#'
#' @references
#' Kabali C (2025). AppRaise: Software for quantifying evidence uncertainty
#' in systematic reviews using a posterior mixture model. *Journal of
#' Evaluation in Clinical Practice*, 31, 1-12. https://doi.org/10.1111/jep.70272.
#'
#' @examples
#' # Simulated posterior draws
#' set.seed(123)
#' mid_samples <- rnorm(2000, mean = -0.3, sd = 0.1)
#'
#' # Posterior probability of benefit
#' posterior_probability(mid_samples)
#'
#' @export
posterior_probability <- function(mid_samples) {
  if (!is.numeric(mid_samples)) {
    stop("mid_samples must be numeric", call. = FALSE)
  }
  mean(mid_samples)
}

#' Posterior summary statistics
#'
#' Computes posterior mean and credible interval
#'
#' @param theta_samples Numeric vector of posterior draws
#' @param probs Credible interval probabilities
#'
#' @return Named numeric vector
#'
#' @seealso
#' \itemize{
#'   \item \code{\link{posterior_mixture}}
#'   \item \code{vignette("appraise-introduction")}
#' }
#'
#' @references
#' Kabali C (2025). AppRaise: Software for quantifying evidence uncertainty
#' in systematic reviews using a posterior mixture model. *Journal of
#' Evaluation in Clinical Practice*, 31, 1-12. https://doi.org/10.1111/jep.70272
#'
#' @examples
#' # Simulated posterior draws for a treatment effect
#' set.seed(123)
#' theta_samples <- rnorm(2000, mean = -0.4, sd = 0.15)
#'
#' # Posterior mean and 95% credible interval
#' posterior_summary(theta_samples)
#'
#' # Custom credible interval (e.g., 90%)
#' posterior_summary(theta_samples, probs = c(0.05, 0.95))
#'
#' @export
posterior_summary <- function(theta_samples, probs = c(0.025, 0.975)) {

  if (!is.numeric(theta_samples)) {
    stop("theta_samples must be numeric", call. = FALSE)
  }

  c(
    mean = mean(theta_samples),
    stats::quantile(theta_samples, probs = probs)
  )
}


#' Posterior mixture across studies
#'
#' Combines posterior draws across studies using a weighted mixture
#'
#' @param theta_list List of numeric vectors of posterior draws
#' @param weights Numeric vector of study weights
#'
#' @return A list containing mixture draws and summaries
#'
#' @seealso
#' \itemize{
#'   \item \code{\link{posterior_probability}}
#'   \item \code{\link{posterior_summary}}
#'   \item \code{vignette("appraise-introduction")}
#' }

#'
#' @references
#' Kabali C (2025). AppRaise: Software for quantifying evidence uncertainty
#' in systematic reviews using a posterior mixture model. *Journal of
#' Evaluation in Clinical Practice*, 31, 1-12. https://doi.org/10.1111/jep.70272.
#'
#' @examples
#' # Simulate posterior draws from two studies
#' theta1 <- rnorm(1000, mean = -0.6, sd = 0.1)
#' theta2 <- rnorm(1000, mean = -0.4, sd = 0.15)
#'
#' # Combine using relevance weights
#' mix <- posterior_mixture(
#'   theta_list = list(theta1, theta2),
#'   weights = c(0.6, 0.4)
#' )
#'
#' # Mixture draws
#' head(mix$draws)
#'
#' # Posterior summary (mean and 95% credible interval)
#' mix$summary
#' @export
posterior_mixture <- function(theta_list, weights) {

  if (!is.list(theta_list) || !all(vapply(theta_list, is.numeric, logical(1)))) {
    stop("theta_list must be a list of numeric vectors", call. = FALSE)
  }

  if (!is.numeric(weights) || length(weights) != length(theta_list)) {
    stop("weights must be numeric and same length as theta_list", call. = FALSE)
  }

  weights <- weights / sum(weights)

  rep_counts <- pmax(round(weights * 10), 1)

  mixture_draws <- unlist(
    mapply(
      function(samples, count) rep(samples, count),
      theta_list,
      rep_counts,
      SIMPLIFY = FALSE
    )
  )

  list(
    draws = mixture_draws,
    summary = posterior_summary(mixture_draws)
  )
}
