% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simList-accessors.R
\name{outputs}
\alias{outputs}
\alias{outputs,simList-method}
\alias{outputs<-}
\alias{outputs<-,simList-method}
\alias{registerOutputs}
\alias{outputArgs}
\alias{outputArgs,simList-method}
\alias{outputArgs<-}
\alias{outputArgs<-,simList-method}
\title{Simulation outputs}
\usage{
outputs(sim)

\S4method{outputs}{simList}(sim)

outputs(sim) <- value

\S4method{outputs}{simList}(sim) <- value

registerOutputs(filename, sim, ...)

outputArgs(sim)

\S4method{outputArgs}{simList}(sim)

outputArgs(sim) <- value

\S4method{outputArgs}{simList}(sim) <- value
}
\arguments{
\item{sim}{A \code{simList}. If missing, then the function will search in the call
stack, so it will find it if it is in a \code{SpaDES} module.}

\item{value}{The object to be stored at the slot. See Details.}

\item{filename}{The filename to register in the \code{outputs(sim)} data.frame. If
missing, an attempt will be made to search for either a \code{file} or \code{filename}
argument in the call itself. This means that this function can be used with
the pipe, as long as the returned return from the upstream pipe function is
a filename or if it is \code{NULL} (e.g., \code{saveRDS}), then it will find the \code{file}
argument and use that.}

\item{...}{Not used.}
}
\value{
A \code{simList} which will be the \code{sim} passed in with a new object registered
in the \code{outputs(sim)}
}
\description{
Accessor functions for the \code{outputs} slots in a \code{simList} object.

If a module saves a file to disk during events, it can be useful to keep track
of the files that are saved e.g., for \code{\link[=saveSimList]{saveSimList()}} so that all files can
be added to the archive. In addition to setting \code{outputs} at the \code{simInit}
stage, a module developer can also put this in a using any saving mechanism that
is relevant (e.g., \code{qs2::qs_save}, \code{saveRDS} etc.). When a module event does this
it can be useful to register that saved file. \code{registerOutputs} offers an additional
mechanism to do this. See examples.
}
\details{
These functions are one of three mechanisms to add information about which output files to save.
\enumerate{
\item As arguments to a \code{simInit} call. Specifically, \code{inputs} or \code{outputs}.
See \code{?simInit}.
\item With the \code{outputs(simList)} function call.
\item By adding a function called \code{.inputObjects} inside a module, which will be executed
during the \code{simInit} call. This last way is the most "modular" way to create
default data sets for your model.
}

See below for more details.

Note using \code{registerOutputs}: a user can pass any other
arguments to \code{registerOutputs} that are in the
\code{outputs(sim)} data.frame, such as \code{objectName}, \code{fun}, \code{package}, though these
will not be used to save the files as this function is only about
registering an output that has already been saved.
}
\note{
The automatic file type handling only adds the correct extension from a given
\code{fun} and \code{package}. It does not do the inverse, from a given extension find the
correct \code{fun} and \code{package}.
}
\section{outputs function or argument in \code{simInit}}{


\code{outputs} accepts a data.frame similar to the \code{inputs} data.frame, but
with up to 6 columns.

\tabular{ll}{
\code{objectName} \tab required, character string indicating the name of the object
in the \code{simList} that will be saved to disk (without the \verb{sim$} prefix).\cr

\code{file} \tab optional, a character string indicating the file path to save to.
The default is to concatenate \code{objectName} with the model timeunit and
\code{saveTime}, separated by underscore, '\verb{_}'. So a default filename would be
\code{"Fires_year1.rds"}.\cr

\code{fun} \tab optional, a character string indicating the function to use to
save that file. The default is \code{\link[=saveRDS]{saveRDS()}} \cr

\code{package} \tab optional character string indicating the package in
which to find the \code{fun});\cr

\code{saveTime} \tab optional numeric, indicating when in simulation time the file
should be saved. The default is the lowest priority at \code{end(sim)},
i.e., at the very end. \cr

\code{arguments} \tab is a list of lists of named arguments, one list for each
\code{fun}. For example, if \code{fun = "write.csv"},
\code{arguments = list(row.names = TRUE)} will pass the argument
\code{row.names = TRUE} to \code{write.csv}  If there is only one list,
then it is assumed to apply to all files and will be recycled as per normal R
rules of recycling for each \code{fun}.\cr
}

See the modules vignette for more details (\code{browseVignettes("SpaDES.core")}).
}

\examples{
#######################
# outputs
#######################

tmpdir <- file.path(tempdir(), "outputs") |> checkPath(create = TRUE)
tmpFile <- file.path(tmpdir, "temp.rds")
tempObj <- 1:10

# Can add data.frame of outputs directly into simInit call
sim <- simInit(objects = c("tempObj"),
               outputs = data.frame(objectName = "tempObj"),
               paths = list(outputPath = tmpdir))
outputs(sim) # To see what will be saved, when, what filename
sim <- spades(sim)
outputs(sim) # To see that it was saved, when, what filename

# Also can add using assignment after a simList object has been made
sim <- simInit(objects = c("tempObj"), paths = list(outputPath = tmpdir))
outputs(sim) <- data.frame(objectName = "tempObj", saveTime = 1:10)
sim <- spades(sim)
outputs(sim) # To see that it was saved, when, what filename.

# can do highly variable saving
tempObj2 <- paste("val", 1:10)
df1 <- data.frame(col1 = tempObj, col2 = tempObj2)
sim <- simInit(objects = c("tempObj", "tempObj2", "df1"),
  paths = list(outputPath = tmpdir))
outputs(sim) <- data.frame(
     objectName = c(rep("tempObj", 2), rep("tempObj2", 3), "df1"),
     saveTime = c(c(1, 4), c(2, 6, 7), end(sim)),
     fun = c(rep("saveRDS", 5), "write.csv"),
     package = c(rep("base", 5), "utils"),
     stringsAsFactors = FALSE)
# since write.csv has a default of adding a column, x, with rownames, must add additional
#   argument for 6th row in data.frame (corresponding to the write.csv function)
outputArgs(sim)[[6]] <- list(row.names = FALSE)
sim <- spades(sim)
outputs(sim)

# read one back in just to test it all worked as planned
newObj <- read.csv(dir(tmpdir, pattern = "year10.csv", full.name = TRUE))
newObj

# using saving with SpaDES-aware methods
# To see current ones SpaDES can do
.saveFileExtensions()

library(terra)
ras <- rast(ncol = 4, nrow = 5)
ras[] <- 1:20

sim <- simInit(objects = c("ras"), paths = list(outputPath = tmpdir))
outputs(sim) <- data.frame(
  file = "test",
  fun = "writeRaster",
  package = "terra",
  objectName = "ras",
  stringsAsFactors = FALSE)

simOut <- spades(sim)
outputs(simOut)
newRas <- rast(dir(tmpdir, full.name = TRUE, pattern = ".tif")[1])
all.equal(newRas, ras) # Should be TRUE
# Clean up after
unlink(tmpdir, recursive = TRUE)
# For `registerOutputs`
sim <- simInit()
# This would normally be a save call, e.g., `writeRaster`
tf <- reproducible::tempfile2(fileext = ".tif")
sim <- registerOutputs(sim, filename = tf)

# Using a pipe
tf <- reproducible::tempfile2(fileext = ".rds")
sim$a <- 1
sim <- saveRDS(sim$a, tf) |> registerOutputs()
# confirm:
outputs(sim) # has object --> saved = TRUE

}
\seealso{
\code{\link[=registerOutputs]{registerOutputs()}} which enables files that are saved to be added to
the \code{simList} using the \code{outputs(sim)} mechanism, so the files that are saved
during a module event can be tracked at the \code{simList} level. \code{\link[=saveSimList]{saveSimList()}}
which will optionally add all the outputs that are tracked into an archive.

\code{\link[=Plots]{Plots()}}, \code{\link[=outputs]{outputs()}}
}
