#' Compute Scalable Mutation-Rate Vector
#'
#' Implements the three mutation-rate strategies (ScaI, ScaII, ScaIII)
#' described in the reference paper.
#'
#' @param method Character, one of "ScaI", "ScaII", "ScaIII".
#' @param beta_star Target parameter vector (binary or factor).
#' @param beta_init Current/initial parameter vector.
#' @param L Integer > 0, length of the parameter vector.
#' @param N_chain Integer > 1, number of parallel chains.
#'
#' @return List with components:
#' \item{Q}{Mutation-rate vector (length 2, r, or N_chain).}
#' \item{Q0}{Base mutation rate (scalar).}
#' \item{r}{Scalar, present only for ScaII - number of active components.}
#' \item{w}{Vector, present only for ScaIII - normalized random weights.}
#'
#' @examples
#' beta_star <- c(1, 0, 1, 1, 0)
#' beta_init <- c(1, 1, 1, 0, 0)
#' compute_mutation_rate("ScaII", beta_star, beta_init, L = 5, N_chain = 8)
#'
#' @export
compute_mutation_rate <- function(method = c("ScaI", "ScaII", "ScaIII"),
                                  beta_star, beta_init, L, N_chain) {
  method <- match.arg(method)
  dH <- hamming_distance(beta_star, beta_init)   # normalized Hamming
  Q0 <- min(3 * dH / L, 1)                       # eq. (2.3) Q_P^0
  
  switch(method,
         ScaI = {
           tau <- stats::runif(1)
           Qs <- c(tau, 1 - tau) * Q0
           list(Q = Qs, Q0 = Q0)
         },
         ScaII = {
           r <- sample(3:(N_chain - 1), 1)
           w <- rep(1 / r, r)
           Qs <- w * Q0
           list(Q = Qs, Q0 = Q0, r = r)
         },
         ScaIII = {
           w <- stats::runif(N_chain)
           w <- w / sum(w)
           Qs <- w * Q0
           list(Q = Qs, Q0 = Q0, w = w)
         })
}
