#' Simulates Network Dynamics under Iterative Perturbations
#'
#' This function simulates the behavior of a network undergoing multiple iterations of custom perturbations.
#' Starting with an initial healthy network, it calculates the network's trajectory by solving the system of
#' ordinary differential equations (ODEs) after each perturbation. Perturbations are applied iteratively, followed by
#' dimension reduction using the specified reduction function.
#'
#'
#' @param system A function defining the system's dynamics: `system(time, x, params)` which returns a list of state deltas `dx`.
#' @param M The initial adjacency matrix of the network.
#' @param x0 Initial conditions of the network's nodes (numeric vector).
#' @param perturbation_function A function defining the perturbation applied to the network. Its signature should be
#'        `perturbation(params, iter)`, where `params` are the current parameters passed to `system`, and `iter` is the current iteration.
#'        Access the current network state via `params$M`. It should return updated parameters or `NULL` if no further perturbations should be applied.
#' @param initial_parameter_function A function of type `f(M) -> list` that takes the adjacency matrix of the network as input and returns the initial parameters of the system.
#' @param times A vector of time points for the ODE solver.
#' @param reduction A reduction function applied to the ODE solution. This can be `identity` (for all node states) or functions like `mean` or `median`. The function signature should be either `f(numeric) -> numeric` or `f(matrix) -> matrix`, depending on whether `only.final.state` is `TRUE` or `FALSE`.
#' @param to.numeric Logical; if `TRUE`, converts the final list to a numeric matrix. If `FALSE`, returns a list `L`, where `L[[i]]` is the network state after `i` perturbations, post-reduction.
#' @param only.final.state Logical; if `TRUE`, applies reduction only to the final state of the system. If `FALSE`, the reduction function is applied to the full ODE trajectory, which can be memory-intensive.
#'
#' @return Depending on `to.numeric`, returns either a list or a numeric matrix, representing the system's state across multiple perturbations.
#' If `to.numeric` is `FALSE`, the function returns a list `L`, where `L[[i]]`
#' represents the final state of the system after the `i`-th perturbation. If `TRUE`, the list is converted to a
#' numeric matrix before being returned.
#' @import deSolve
#' @export
#' @examples
#' \donttest{
#'    node_file <- system.file("extdata", "IL17.nodes.csv", package = "Rato")
#'    edge_file <- system.file("extdata", "IL17.edges.csv", package = "Rato")
#'    g <- Rato::graph.from.csv(node_file, edge_file, sep=",", header=TRUE)
#'  
#'    initial_params = list('f' = 1, 'h'=2, 'B'=0.01)
#'    removal_order = NULL
#'    update_params = identity
#'    reduction = identity
#'    initial_parameter_function <- function(M) {
#'
#'      if (is.list(initial_params)) {
#'        params <- initial_params
#'      }
#'      else if (is.function(initial_params)) {
#'        params <- initial_params(M)
#'      }
#'      n <- nrow(M) # The number of nodes of the graph
#'
#'      if(is.null(removal_order)){
#'          removal_order <- sample(1:n, n) 
#'      }
#'
#'      if(is.null(params$M)){
#'        params$M = M
#'      } 
#'      params$removal_order = removal_order
#'      return(params)
#'    }
#'    perturbation <- function(params, iter) {
#'      removal_order <- params$removal_order
#'      M <- params$M
#'  
#'      if(length(removal_order) > 0 ) {
#'        index <- removal_order[1] 
#'        removal_order <- removal_order[-1] 
#'        M[index, ] <- 0 # Remove entries
#'        M[, index] <- 0 # Remove entries
#'  
#'        params$M = M
#'        params$removal_order = removal_order
#'  
#'        # Add the list of removed indices to the parameter list.
#'        if(is.null(params$removed_indices)){
#'          params$removed_indices <- c(index)
#'        } else {
#'          removed_indices <- c(params$removed_indices, index)
#'          params$removed_indices <- removed_indices
#'        }
#'  
#'        output <- update_params(params)
#'        return(output) # Return updated parameters
#'      }
#'  
#'      # If we removed every single entry, just STOP.
#'      return(NULL)
#'    }
#'  
#'    Rato::perturbation.thread( Rato::Michaelis.Menten
#'                              , g$M
#'                              , g$initial_values
#'                              , initial_parameter_function = initial_parameter_function
#'                              , perturbation_function = perturbation)
#'
#' }


perturbation.thread <- function( system
                                 , M
                                 , x0
                                 , perturbation_function
                                 , initial_parameter_function
                                 , times = seq(0, 100, 1) # Idk about this
                                 , reduction = identity
                                 , to.numeric = FALSE
                                 , only.final.state = TRUE # Whether reduction applies on the final state of the trajectory, or in all of the points. This should always be set to true. Only set it to false if you absolutely know what you are doing. It will consume a LOT of memory.
                                 ) {
  # Get the initial parameters for this specific matrix
  param <- initial_parameter_function(M)

  # This is the expression of the system as the perturbation happens.
  # This will become the output of the function
  expressions <- list()

  iter <- 1

  # We iterate for as long as the permutation function returns something valid.
  # The convention is: To stop, simply return NA from the permutation function.
  # This follows a C-like strategy which is not the most elegant, but R sucks so it does not matter.
  while(!is.null(param)) {

    # The initial point can be fixed, or a function of the matrix.
    if(is.function(x0)) {
      y <- x0(param)
    } else {
      y <- x0
    }

    # Run the dynamics
    current_trajectory <- deSolve::ode(func = system, y = y, times=times, parms=param)

    if(only.final.state) {
      final_state <- current_trajectory[dim(current_trajectory)[1], -1]
    } else {
      final_state <- current_trajectory
    }

    # We may want to store the expression of every node.
    # Or perhaps, we may only want to store the overall mean expression.
    # The user can use the reduction function to transform the output of the deSolve into something useful.
    this_expression <- reduction(final_state)

    # Save the final expression
    expressions[[iter]] <- this_expression

    # Perform the perturbation and update the current iteration
    param <- perturbation_function(param, iter)
    iter <- iter + 1
  }

  # If the user wants to convert the resulting list to a vector ( Useful where you use a reduction whose
  # output is one dimensional. Then your final output is a simple vector of numeric values )
  if(to.numeric) {
    return(as.numeric(expressions))
  }
  # Otherwise, just return the entire list (Useful for when you want to look at the specific expression of a
  # single node throughout the perturbation process)
  else {
    return(expressions)
  }
}
