# Tests for qq_regression function

test_that("qq_regression returns correct class", {
  set.seed(42)
  n <- 100
  x <- rnorm(n)
  y <- 0.5 * x + rnorm(n, sd = 0.5)
  
  result <- qq_regression(y, x, 
                          y_quantiles = seq(0.2, 0.8, by = 0.2),
                          x_quantiles = seq(0.2, 0.8, by = 0.2),
                          verbose = FALSE)
  
  expect_s3_class(result, "qq_regression")
})

test_that("qq_regression results have correct structure", {
  set.seed(42)
  n <- 100
  x <- rnorm(n)
  y <- 0.5 * x + rnorm(n, sd = 0.5)
  
  y_q <- seq(0.2, 0.8, by = 0.2)
  x_q <- seq(0.2, 0.8, by = 0.2)
  
  result <- qq_regression(y, x, 
                          y_quantiles = y_q,
                          x_quantiles = x_q,
                          verbose = FALSE)
  
  # Check result components
  expect_true("results" %in% names(result))
  expect_true("y_quantiles" %in% names(result))
  expect_true("x_quantiles" %in% names(result))
  expect_true("n_obs" %in% names(result))
  
  # Check results data frame
  expect_true("y_quantile" %in% names(result$results))
  expect_true("x_quantile" %in% names(result$results))
  expect_true("coefficient" %in% names(result$results))
  expect_true("p_value" %in% names(result$results))
  expect_true("r_squared" %in% names(result$results))
  
  # Check dimensions
  expect_equal(nrow(result$results), length(y_q) * length(x_q))
})

test_that("qq_regression handles missing values", {
  set.seed(42)
  n <- 100
  x <- rnorm(n)
  y <- 0.5 * x + rnorm(n, sd = 0.5)
  
  # Add some missing values
  y[c(1, 5, 10)] <- NA
  x[c(2, 6, 11)] <- NA
  
  result <- qq_regression(y, x, 
                          y_quantiles = c(0.25, 0.5, 0.75),
                          x_quantiles = c(0.25, 0.5, 0.75),
                          verbose = FALSE)
  
  expect_s3_class(result, "qq_regression")
  expect_equal(result$n_obs, sum(complete.cases(y, x)))
})

test_that("qq_regression validates inputs", {
  set.seed(42)
  x <- rnorm(100)
  y <- rnorm(100)
  
  # Wrong input types
  expect_error(qq_regression("a", x, verbose = FALSE))
  expect_error(qq_regression(y, "b", verbose = FALSE))
  
  # Length mismatch
  expect_error(qq_regression(y, x[1:50], verbose = FALSE))
  
  # Invalid quantiles
  expect_error(qq_regression(y, x, y_quantiles = c(-0.1, 0.5), verbose = FALSE))
  expect_error(qq_regression(y, x, x_quantiles = c(0.5, 1.1), verbose = FALSE))
})

test_that("qq_regression respects verbose option", {
  set.seed(42)
  x <- rnorm(50)
  y <- rnorm(50)
  
  # Should produce no messages when verbose = FALSE
  expect_silent(
    qq_regression(y, x, 
                  y_quantiles = c(0.25, 0.75),
                  x_quantiles = c(0.25, 0.75),
                  verbose = FALSE)
  )
})

test_that("print method works", {
  set.seed(42)
  x <- rnorm(50)
  y <- rnorm(50)
  
  result <- qq_regression(y, x, 
                          y_quantiles = c(0.25, 0.75),
                          x_quantiles = c(0.25, 0.75),
                          verbose = FALSE)
  
  expect_output(print(result), "Quantile-on-Quantile")
})

test_that("summary method works", {
  set.seed(42)
  x <- rnorm(50)
  y <- rnorm(50)
  
  result <- qq_regression(y, x, 
                          y_quantiles = c(0.25, 0.75),
                          x_quantiles = c(0.25, 0.75),
                          verbose = FALSE)
  
  expect_output(summary(result), "Coefficient Statistics")
})

test_that("qq_statistics returns correct structure", {
  set.seed(42)
  x <- rnorm(50)
  y <- rnorm(50)
  
  result <- qq_regression(y, x, 
                          y_quantiles = c(0.25, 0.75),
                          x_quantiles = c(0.25, 0.75),
                          verbose = FALSE)
  
  stats <- qq_statistics(result)
  
  expect_s3_class(stats, "data.frame")
  expect_true("Statistic" %in% names(stats))
  expect_true("Value" %in% names(stats))
})

test_that("qq_to_matrix returns correct dimensions", {
  set.seed(42)
  x <- rnorm(50)
  y <- rnorm(50)
  
  y_q <- c(0.25, 0.5, 0.75)
  x_q <- c(0.25, 0.5, 0.75)
  
  result <- qq_regression(y, x, 
                          y_quantiles = y_q,
                          x_quantiles = x_q,
                          verbose = FALSE)
  
  mat <- qq_to_matrix(result, type = "coefficient")
  
  expect_equal(nrow(mat), length(y_q))
  expect_equal(ncol(mat), length(x_q))
})
