# PeruAPIs - Access Peruvian Data via Public APIs and Curated Datasets
# Version 0.1.0
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_peru_population


library(testthat)

test_that("get_peru_population() returns a tibble with correct structure and types", {
  skip_on_cran()
  result <- get_peru_population()

  # Structure checks
  expect_s3_class(result, "tbl_df")
  expect_named(result, c("indicator", "country", "year", "value", "value_label"))

  # Type checks
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_true(is.numeric(result$value) || is.integer(result$value))
  expect_type(result$value_label, "character")
})

test_that("get_peru_population() returns only Peru data", {
  skip_on_cran()
  result <- get_peru_population()
  expect_true(all(result$country == "Peru"))
})

test_that("get_peru_population() returns years from 2010 to 2022", {
  skip_on_cran()
  result <- get_peru_population()
  expect_true(all(result$year %in% 2010:2022))
})

test_that("get_peru_population() returns exactly 13 rows", {
  skip_on_cran()
  result <- get_peru_population()
  expect_equal(nrow(result), 13)
})

test_that("get_peru_population() includes the correct indicator label", {
  skip_on_cran()
  result <- get_peru_population()
  expect_true(all(result$indicator == "Population, total"))
})

test_that("get_peru_population(): no NA values in year column", {
  skip_on_cran()
  result <- get_peru_population()
  expect_false(any(is.na(result$year)))
})

test_that("get_peru_population(): value column is numeric and positive", {
  skip_on_cran()
  result <- get_peru_population()
  expect_true(all(is.finite(result$value)))
  expect_true(all(result$value > 0))
})

test_that("get_peru_population(): value_label is formatted with commas", {
  skip_on_cran()
  result <- get_peru_population()
  expect_true(all(grepl(",", result$value_label)))
})

test_that("get_peru_population(): years are sorted in descending order", {
  skip_on_cran()
  result <- get_peru_population()
  expect_equal(result$year, sort(result$year, decreasing = TRUE))
})

test_that("get_peru_population(): indicator is consistent across rows", {
  skip_on_cran()
  result <- get_peru_population()
  expect_equal(length(unique(result$indicator)), 1)
})

test_that("get_peru_population(): country is consistent across rows", {
  skip_on_cran()
  result <- get_peru_population()
  expect_equal(length(unique(result$country)), 1)
})
