# HCNHE.R
#' Half-Cauchy NHE Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Half-Cauchy NHE distribution.
#'
#' The Half-Cauchy NHE distribution is parameterized by the parameters
#' \eqn{\beta > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Half-Cauchy NHE distribution has CDF:
#'
#' \deqn{
#' F(x; \beta, \lambda, \theta) =
#' \frac{2}{\pi}
#' \arctan\left\{
#' -\frac{1}{\theta}
#' \left( 1 - (1 + \lambda x)^{\beta} \right)
#' \right\},
#' \quad x > 0.
#' }
#'
#' where \eqn{\beta}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dhc.NHE()} — Density function
#'   \item \code{phc.NHE()} — Distribution function
#'   \item \code{qhc.NHE()} — Quantile function
#'   \item \code{rhc.NHE()} — Random generation
#'   \item \code{hhc.NHE()} — Hazard function
#' }
#'
#' @usage
#' dhc.NHE(x, beta, lambda, theta, log = FALSE)
#' phc.NHE(q, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qhc.NHE(p, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rhc.NHE(n, beta, lambda, theta)
#' hhc.NHE(x, beta, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dhc.NHE}: numeric vector of (log-)densities
#'   \item \code{phc.NHE}: numeric vector of probabilities
#'   \item \code{qhc.NHE}: numeric vector of quantiles
#'   \item \code{rhc.NHE}: numeric vector of random variates
#'   \item \code{hhc.NHE}: numeric vector of hazard values
#' }
#'
#' @references
#' Chaudhary, A. K., & Kumar, V.(2021). 
#' Arctan Exponential Extension Distribution with Properties and Applications. 
#' \emph{International Journal of Applied Research (IJAR)}, \bold{7(1)}, 432--442. 
#' \doi{10.22271/allresearch.2021.v7.i1f.8251}
#' 
#' Telee, L. B. S., & Kumar, V. (2022). 
#' Some properties and applications of half-Cauchy exponential extension 
#' distribution.\emph{ Int. J. Stat. Appl. Math.}, \bold{7(6)}, 91--101. 
#' \doi{10.22271/maths.2022.v7.i6b.902} 
#'
#' Kumar, V. (2010). Bayesian analysis of exponential extension model. 
#' \emph{J. Nat. Acad. Math.}, \bold{24}, 109-128.
#'
#' @examples
#' x <- seq(1.0, 5, 0.25)
#' dhc.NHE(x, 2.0, 0.5, 2.5)
#' phc.NHE(x, 2.0, 0.5, 2.5)
#' qhc.NHE(0.5, 2.0, 0.5, 2.5)
#' rhc.NHE(10, 2.0, 0.5, 2.5)
#' hhc.NHE(x, 2.0, 0.5, 2.5)
#' 
#' # Data
#' x <- stress66
#' # ML estimates
#' params = list(beta=95.2115, lambda=0.0184, theta=118.0656)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = phc.NHE, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qhc.NHE, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dhc.NHE, pfun=phc.NHE, plot=TRUE)
#' print.gofic(out)
#' 
#' @name HCNHE
#' @aliases dhc.NHE phc.NHE qhc.NHE rhc.NHE hhc.NHE
#' @export
NULL

#' @rdname HCNHE
#' @usage NULL
#' @export   
dhc.NHE <- function (x, beta, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, beta, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(beta, lambda, theta) <= 0)) {
        stop("Parameters 'beta', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- (1 + lambda * x[valid])	
	z <- -(1-u^beta)/theta	
    pdf[valid] <- (2/pi)*(beta*lambda/theta)* (u^(beta-1)) * (1/(1+ (z^2)))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname HCNHE
#' @usage NULL
#' @export
phc.NHE <- function (q, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, beta, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(beta, lambda, theta) <= 0)) {
		stop("Parameters 'beta', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- -(1/theta)* (1 - (1 + lambda * q[valid]) ^ beta)
    cdf[valid] <- (2/pi) * atan(u)  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname HCNHE
#' @usage NULL
#' @export
qhc.NHE <- function (p, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, beta, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(beta, lambda, theta) <= 0))
		stop("Invalid parameters: beta, 'lambda' and 'theta' must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	qtl[valid] <-  (1/lambda)*((1+ theta*tan(p[valid] * pi *0.5)) ^ (1/beta) -1)	   
    }    
    return(qtl)  
 }  

#' @rdname HCNHE
#' @usage NULL
#' @export
rhc.NHE <- function(n, beta, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(beta, lambda, theta), is.numeric)))
		stop("'beta', 'lambda', and 'theta' must be numeric.")
	if (any(c(beta, lambda, theta) <= 0))
		stop("Invalid parameters: beta, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qhc.NHE(u, beta, lambda, theta)
    return(q)
}

#' @rdname HCNHE
#' @usage NULL
#' @export
hhc.NHE <- function (x, beta, lambda, theta)
{
	if (!all(sapply(list(x, beta, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(beta, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")    
    nume <- dhc.NHE(x, beta, lambda, theta)
	surv <- 1.0 - phc.NHE(x, beta, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf) 
}





