% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcee.R
\name{mcee}
\alias{mcee}
\title{Mediated Causal Excursion Effects for MRTs (streamlined)}
\usage{
mcee(
  data,
  id,
  dp,
  outcome,
  treatment,
  mediator,
  availability = NULL,
  rand_prob,
  time_varying_effect_form,
  control_formula_with_mediator,
  control_reg_method = c("glm", "gam", "rf", "ranger", "sl"),
  weight_per_row = NULL,
  specific_dp_only = NULL,
  verbose = TRUE,
  SL.library = NULL
)
}
\arguments{
\item{data}{A data.frame in long format (one row per id-by-decision point).}

\item{id}{Character. Column name for subject identifier.}

\item{dp}{Character. Column name for decision point index (must increase strictly within subject).}

\item{outcome}{Character. Column name for distal outcome (constant within subject).}

\item{treatment}{Character. Column name for treatment (coded 0/1).}

\item{mediator}{Character. Column name for mediator.}

\item{availability}{Optional character. Column name for availability (0/1). If \code{NULL}, all rows are treated as available.}

\item{rand_prob}{Either a column name in \code{data} or a scalar giving the known randomization probability \eqn{P(A_t=1 \mid H_t)}.  (Technically,
this is \eqn{P(A_t=I_t\mid H_t)}, but the user is allowed to input \eqn{P(A_t=1\mid H_t)}
and the function will automatically correct it by setting \code{p1 = 1} when \eqn{I_t = 0}.)}

\item{time_varying_effect_form}{RHS-only formula for the basis \eqn{f(t)} (e.g., \code{~ 1}, \code{~ dp}, \code{~ poly(dp,2)}).}

\item{control_formula_with_mediator}{RHS-only formula for control variables used in nuisance models that may include the mediator (the wrapper will drop the mediator internally for nuisances that must exclude it).}

\item{control_reg_method}{Learner for nuisance fits: one of \code{"glm"}, \code{"gam"}, \code{"rf"}, \code{"ranger"}, \code{"sl"}.}

\item{weight_per_row}{Optional numeric vector of row weights (nonnegative, length \code{nrow(data)}). If \code{NULL}, uniform within-id weights are used.}

\item{specific_dp_only}{Optional numeric vector of decision points to target; internally converted to \code{weight_per_row}.}

\item{verbose}{Logical; print progress messages.}

\item{SL.library}{Optional character vector of SuperLearner libraries (used when \code{control_reg_method = "sl"}).}
}
\value{
An object of class \code{"mcee_fit"} with elements:
\itemize{
  \item \code{mcee_fit}: list with \code{alpha_hat}, \code{beta_hat}, \code{alpha_se}, \code{beta_se},
        \code{varcov}, \code{alpha_varcov}, \code{beta_varcov}.
  \item \code{nuisance_models}: fitted Stage-1 models for \code{p,q,eta,mu,nu}.
  \item \code{nuisance_fitted}: per-row fitted values for the nuisance functions.
  \item \code{meta}: list with basis dimension, number of ids, per-id lengths, weights used.
  \item \code{call}: the matched call.
}
}
\description{
Estimates the Natural Direct Excursion Effect (NDEE; \eqn{\alpha}) and
Natural Indirect Excursion Effect (NIEE; \eqn{\beta}) for distal outcomes
in micro-randomized trials (MRTs). Assumes the randomization probability
is known (via \code{rand_prob}) and fits all nuisance functions using the
same learner specified by \code{control_reg_method}.
}
\details{
Requirements: rows grouped by subject, strictly increasing \code{dp} within subject,
no missing (\code{NA}/\code{NaN}/\code{Inf}) in relevant variables. If \code{availability}
is supplied, the wrapper enforces at \eqn{I=0}: \eqn{p_1=q_1=1} in the nuisances.
}
\examples{
set.seed(1)
n <- 10
T <- 4
id <- rep(1:n, each = T)
dp <- rep(1:T, times = n)
A <- rbinom(n * T, 1, 0.5)
M <- rbinom(n * T, 1, plogis(-0.2 + 0.3 * A + 0.1 * dp))
Y <- ave(0.5 * A + 0.6 * M + 0.1 * dp + rnorm(n * T), id)
dat <- data.frame(id, dp, A, M, Y)

fit <- mcee(dat, "id", "dp", "Y", "A", "M",
    time_varying_effect_form = ~1,
    control_formula_with_mediator = ~ dp + M,
    control_reg_method = "glm",
    rand_prob = 0.5, verbose = TRUE
)
summary(fit)

}
\seealso{
\code{\link{summary.mcee_fit}}, \code{\link{mcee_general}}, \code{\link{mcee_userfit_nuisance}}
}
