#' tvm.imputation: Impute Influential Responses in Longitudinal Data
#'
#' This function identifies influential response values using the `tvm` function, replaces them with NA, and imputes the missing values using the `mice` package.
#'
#' @param data A data frame containing the longitudinal data.
#' @param subject_col Character. The name of the column representing subject IDs.
#' @param time_col Character. The name of the column representing time points.
#' @param response_col Character. The name of the column representing the response variable.
#' @param k Numeric. The number of clusters for the `tvm` function.
#' @param impute_method Character. The imputation method to be used in `mice` (default is "pmm").
#' @param m Numeric. The number of multiple imputations to be performed (default is 5).
#'
#' @return A data frame with imputed values for the influential response points while maintaining original NA values.
#'
#' @importFrom mice mice complete
#' @importFrom dplyr mutate
#' @export
#'
#' @examples
#' infsdata <- infsdata[1:5,]
#' imptvm <- tvm.imputation(infsdata, "subject_id", "time", "response", k = 3)
#' head(imptvm)

tvm.imputation <- function(data,
                           subject_col,
                           time_col,
                           response_col,
                           k,
                           impute_method = "pmm",
                           m = 5) {

  # Store original NA positions
  na_positions <- is.na(data[[response_col]])

  # Identify influential time points using tvm
  result <- tvm(data, subject_col, time_col, response_col, k)
  influential_data <- result$influential_time_data

  # If no influential points are found, return original data
  if (is.null(influential_data) || nrow(influential_data) == 0) {
    warning("No influential time points detected; returning original data.")
    return(data)
  }

  # Replace influential response values with NA (type-safe)
  key_all <- paste(data[[subject_col]], data[[time_col]])
  key_inf <- paste(influential_data[[subject_col]],
                   influential_data[[time_col]])

  data[[response_col]] <- dplyr::if_else(
    key_all %in% key_inf,
    NA_real_,
    data[[response_col]]
  )

  # Perform imputation (silenced for CRAN)
  imputed_data <- suppressMessages(
    mice::mice(data, method = impute_method, m = m)
  )

  # Extract first completed dataset
  final_data <- mice::complete(imputed_data, 1)

  # Restore original NA values
  final_data[[response_col]][na_positions] <- NA

  final_data
}
