\name{runEM}
\alias{runEM}
\title{Expectation-Maximization (EM) Algorithm for Gaussian Mixture Models}
\description{
Implements the Expectation-Maximization (EM) algorithm for fitting Gaussian Mixture Models (GMMs).
It iteratively updates the model parameters (means, covariances, and mixing proportions) until convergence.
}
\usage{
runEM(x, param, max_iter = 100, tol = 1e-5)
}
\arguments{
  \item{x}{A numeric matrix or data frame where rows represent observations and columns represent variables.}
  \item{param}{A list containing the initial parameters for the EM algorithm:
    \itemize{
      \item \code{param[[1]]}: Mixing proportions (\eqn{\pi_k}).
      \item \code{param[[2]]}: Cluster means (\eqn{\mu}).
      \item \code{param[[3]]}: Covariance matrices (\eqn{\Sigma}).
    }
  }
  \item{max_iter}{An integer specifying the maximum number of iterations allowed for the EM algorithm. (default is 100)}
  \item{tol}{A numeric value specifying the convergence tolerance threshold (default is 1e-5). 
  The algorithm stops when the relative change in log-likelihood is below this value.}
}
\value{
A list containing the following components:
  \itemize{
    \item \code{BIC}: Bayesian Information Criterion (BIC) value for model selection.
    \item \code{param}: A list containing the estimated model parameters:
      \itemize{
        \item \code{param[[1]]}: Updated mixing proportions.
        \item \code{param[[2]]}: Updated cluster means.
        \item \code{param[[3]]}: Updated covariance matrices.
      }
    \item \code{cluster_labels}: Cluster assignments (most probable cluster for each observation).
    \item \code{Z}: Posterior probability matrix (\eqn{\gamma}), rounded to 4 decimal places.
  }
}
\details{
The EM algorithm iteratively refines the estimates of the Gaussian Mixture Model (GMM) parameters 
by alternating between two steps:
\itemize{
  \item \strong{E-step}: Computes the posterior probabilities (responsibilities) of cluster membership for each observation.
  \item \strong{M-step}: Updates the parameters (means, covariances, and mixing proportions) based on the computed responsibilities.
}
Convergence is assessed using the log-likelihood function.
}
\seealso{
\code{\link{getInit}}}

\examples{
# Generate synthetic data
set.seed(123)
data <- matrix(rnorm(100 * 2), ncol = 2)

# Initialize parameters using k-means
init_params <- getInit(data, k = 2, method = "kmeans")

# Run the EM algorithm
em_results <- runEM(data, param = init_params, max_iter = 100, tol = 1e-5)

# Print results
print(em_results$BIC)  
print(em_results$cluster)
}
\keyword{EM}
\keyword{clustering}
\keyword{mixture model}
