/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#if !(defined H5B2_FRIEND || defined H5B2_MODULE)
#error "Do not include this file outside the H5B2 package!"
#endif

#ifndef H5B2pkg_H
#define H5B2pkg_H

#include "H5B2private.h"

#include "H5ACprivate.h" 
#include "H5FLprivate.h" 

#define H5B2_SIZEOF_RECORDS_PER_NODE (unsigned)2

#define H5B2_TREE_POINTER_SIZE(sizeof_addr, sizeof_size)                                                     \
    ((sizeof_addr) + H5B2_SIZEOF_RECORDS_PER_NODE + (sizeof_size))

#define H5B2_INT_POINTER_SIZE(h, d)                                                                          \
    ((unsigned)(h)->sizeof_addr                                                   \
     + (h)->max_nrec_size                                                    \
     + (h)->node_info[(d)-1].cum_max_nrec_size                      \
    )

#define H5B2_SIZEOF_CHKSUM 4

#define H5B2_METADATA_PREFIX_SIZE                                                                            \
    ((unsigned)H5_SIZEOF_MAGIC                                                                \
     + (unsigned)1                                                                              \
     + (unsigned)1                                                                            \
     + (unsigned)H5B2_SIZEOF_CHKSUM                                                   \
    )

#define H5B2_HEADER_SIZE(sizeof_addr, sizeof_size)                                                           \
    (                                                                           \
     H5B2_METADATA_PREFIX_SIZE                                                                               \
                                                                                                             \
                                                                                 \
     + (unsigned)4                                                                  \
     + (unsigned)2                                                                \
     + (unsigned)2                                                                        \
     + (unsigned)1                                     \
     + (unsigned)1                                     \
     + H5B2_TREE_POINTER_SIZE(sizeof_addr, sizeof_size)               \
    )

#define H5B2_HEADER_SIZE_FILE(f) (H5B2_HEADER_SIZE(H5F_SIZEOF_ADDR(f), H5F_SIZEOF_SIZE(f)))

#define H5B2_HEADER_SIZE_HDR(h) (H5B2_HEADER_SIZE((h)->sizeof_addr, (h)->sizeof_size))

#define H5B2_INT_PREFIX_SIZE                                                                                 \
    (                                                                           \
     H5B2_METADATA_PREFIX_SIZE                                                                               \
                                                                                                             \
                                                                     \
    )

#define H5B2_LEAF_PREFIX_SIZE                                                                                \
    (                                                                           \
     H5B2_METADATA_PREFIX_SIZE                                                                               \
                                                                                                             \
                                                                     \
    )

#define H5B2_NAT_NREC(b, hdr, idx) ((b) + (hdr)->nat_off[(idx)])

#define H5B2_INT_NREC(i, hdr, idx) H5B2_NAT_NREC((i)->int_native, (hdr), (idx))

#define H5B2_LEAF_NREC(l, hdr, idx) H5B2_NAT_NREC((l)->leaf_native, (hdr), (idx))

#define H5B2_NUM_INT_REC(h, d)                                                                               \
    (((h)->node_size - (H5B2_INT_PREFIX_SIZE + H5B2_INT_POINTER_SIZE(h, d))) /                               \
     ((h)->rrec_size + H5B2_INT_POINTER_SIZE(h, d)))

typedef struct {
    haddr_t  addr;      
    uint16_t node_nrec; 
    hsize_t  all_nrec;  
} H5B2_node_ptr_t;

typedef struct {
    unsigned max_nrec;             
    unsigned split_nrec;           
    unsigned merge_nrec;           
    hsize_t  cum_max_nrec;         
    uint8_t  cum_max_nrec_size;    
    H5FL_fac_head_t *nat_rec_fac;  
    H5FL_fac_head_t *node_ptr_fac; 
} H5B2_node_info_t;

typedef struct H5B2_hdr_t {
    
    H5AC_info_t cache_info;

    
    H5B2_node_ptr_t root; 

    
    uint8_t  split_percent; 
    uint8_t  merge_percent; 
    uint32_t node_size;     
    uint32_t rrec_size;     

    
    uint16_t depth; 

    
    uint8_t max_nrec_size; 

    
    H5F_t            *f;              
    haddr_t           addr;           
    size_t            hdr_size;       
    size_t            rc;             
    size_t            file_rc;        
    bool              pending_delete; 
    uint8_t           sizeof_size;    
    uint8_t           sizeof_addr;    
    H5B2_remove_t     remove_op;      
    void             *remove_op_data; 
    uint8_t          *page;           
    size_t           *nat_off;        
    H5B2_node_info_t *node_info;      
    void             *min_native_rec; 
    void             *max_native_rec; 

    
    bool                swmr_write; 
    H5AC_proxy_entry_t *top_proxy;  
    void               *parent;     
    uint64_t shadow_epoch;          

    
    const H5B2_class_t *cls;    
    void               *cb_ctx; 
} H5B2_hdr_t;

typedef struct H5B2_leaf_t {
    
    H5AC_info_t cache_info;

    
    H5B2_hdr_t *hdr;         
    uint8_t    *leaf_native; 
    uint16_t    nrec;        

    
    H5AC_proxy_entry_t *top_proxy;    
    void               *parent;       
    uint64_t            shadow_epoch; 
} H5B2_leaf_t;

typedef struct H5B2_internal_t {
    
    H5AC_info_t cache_info;

    
    H5B2_hdr_t      *hdr;        
    uint8_t         *int_native; 
    H5B2_node_ptr_t *node_ptrs;  
    uint16_t         nrec;       
    uint16_t         depth;      

    
    H5AC_proxy_entry_t *top_proxy;    
    void               *parent;       
    uint64_t            shadow_epoch; 
} H5B2_internal_t;

struct H5B2_t {
    H5B2_hdr_t *hdr; 
    H5F_t      *f;   
};

typedef enum H5B2_nodepos_t {
    H5B2_POS_ROOT,  
    H5B2_POS_RIGHT, 
    H5B2_POS_LEFT,  
    H5B2_POS_MIDDLE 
} H5B2_nodepos_t;

typedef enum H5B2_update_status_t {
    H5B2_UPDATE_UNKNOWN,          
    H5B2_UPDATE_MODIFY_DONE,      
    H5B2_UPDATE_SHADOW_DONE,      
    H5B2_UPDATE_INSERT_DONE,      
    H5B2_UPDATE_INSERT_CHILD_FULL 
} H5B2_update_status_t;

typedef struct H5B2_hdr_cache_ud_t {
    H5F_t  *f;         
    haddr_t addr;      
    void   *ctx_udata; 
} H5B2_hdr_cache_ud_t;

typedef struct H5B2_internal_cache_ud_t {
    H5F_t      *f;      
    H5B2_hdr_t *hdr;    
    void       *parent; 
    uint16_t    nrec;   
    uint16_t    depth;  
} H5B2_internal_cache_ud_t;

typedef struct H5B2_leaf_cache_ud_t {
    H5F_t      *f;      
    H5B2_hdr_t *hdr;    
    void       *parent; 
    uint16_t    nrec;   
} H5B2_leaf_cache_ud_t;

#ifdef H5B2_TESTING

typedef struct H5B2_node_info_test_t {
    uint16_t depth; 
    uint16_t nrec;  
} H5B2_node_info_test_t;
#endif 

H5FL_EXTERN(H5B2_internal_t);

H5FL_EXTERN(H5B2_leaf_t);

#ifdef H5B2_TESTING
H5_DLLVAR const H5B2_class_t H5B2_TEST[1];
H5_DLLVAR const H5B2_class_t H5B2_TEST2[1];

typedef struct H5B2_test_rec_t {
    hsize_t key; 
    hsize_t val; 
} H5B2_test_rec_t;
#endif 

extern const H5B2_class_t *const H5B2_client_class_g[H5B2_NUM_BTREE_ID];

H5_DLL herr_t H5B2__create_flush_depend(H5AC_info_t *parent_entry, H5AC_info_t *child_entry);
H5_DLL herr_t H5B2__update_flush_depend(H5B2_hdr_t *hdr, unsigned depth, H5B2_node_ptr_t *node_ptr,
                                        void *old_parent, void *new_parent);
H5_DLL herr_t H5B2__destroy_flush_depend(H5AC_info_t *parent_entry, H5AC_info_t *child_entry);

H5_DLL herr_t H5B2__split1(H5B2_hdr_t *hdr, uint16_t depth, H5B2_node_ptr_t *curr_node_ptr,
                           unsigned *parent_cache_info_flags_ptr, H5B2_internal_t *internal,
                           unsigned *internal_flags_ptr, unsigned idx);
H5_DLL herr_t H5B2__redistribute2(H5B2_hdr_t *hdr, uint16_t depth, H5B2_internal_t *internal, unsigned idx);
H5_DLL herr_t H5B2__redistribute3(H5B2_hdr_t *hdr, uint16_t depth, H5B2_internal_t *internal,
                                  unsigned *internal_flags_ptr, unsigned idx);
H5_DLL herr_t H5B2__merge2(H5B2_hdr_t *hdr, uint16_t depth, H5B2_node_ptr_t *curr_node_ptr,
                           unsigned *parent_cache_info_flags_ptr, H5B2_internal_t *internal,
                           unsigned *internal_flags_ptr, unsigned idx);
H5_DLL herr_t H5B2__merge3(H5B2_hdr_t *hdr, uint16_t depth, H5B2_node_ptr_t *curr_node_ptr,
                           unsigned *parent_cache_info_flags_ptr, H5B2_internal_t *internal,
                           unsigned *internal_flags_ptr, unsigned idx);

H5_DLL H5B2_hdr_t *H5B2__hdr_alloc(H5F_t *f);
H5_DLL haddr_t     H5B2__hdr_create(H5F_t *f, const H5B2_create_t *cparam, void *ctx_udata);
H5_DLL herr_t H5B2__hdr_init(H5B2_hdr_t *hdr, const H5B2_create_t *cparam, void *ctx_udata, uint16_t depth);
H5_DLL herr_t H5B2__hdr_incr(H5B2_hdr_t *hdr);
H5_DLL herr_t H5B2__hdr_decr(H5B2_hdr_t *hdr);
H5_DLL herr_t H5B2__hdr_fuse_incr(H5B2_hdr_t *hdr);
H5_DLL size_t H5B2__hdr_fuse_decr(H5B2_hdr_t *hdr);
H5_DLL herr_t H5B2__hdr_dirty(H5B2_hdr_t *hdr);
H5_DLL H5B2_hdr_t *H5B2__hdr_protect(H5F_t *f, haddr_t hdr_addr, void *ctx_udata, unsigned flags);
H5_DLL herr_t      H5B2__hdr_unprotect(H5B2_hdr_t *hdr, unsigned cache_flags);
H5_DLL herr_t      H5B2__hdr_delete(H5B2_hdr_t *hdr);

H5_DLL H5B2_leaf_t *H5B2__protect_leaf(H5B2_hdr_t *hdr, void *parent, H5B2_node_ptr_t *node_ptr, bool shadow,
                                       unsigned flags);
H5_DLL herr_t       H5B2__swap_leaf(H5B2_hdr_t *hdr, uint16_t depth, H5B2_internal_t *internal,
                                    unsigned *internal_flags_ptr, unsigned idx, void *swap_loc);

H5_DLL H5B2_internal_t *H5B2__protect_internal(H5B2_hdr_t *hdr, void *parent, H5B2_node_ptr_t *node_ptr,
                                               uint16_t depth, bool shadow, unsigned flags);

H5_DLL herr_t H5B2__split_root(H5B2_hdr_t *hdr);
H5_DLL herr_t H5B2__create_leaf(H5B2_hdr_t *hdr, void *parent, H5B2_node_ptr_t *node_ptr);
H5_DLL herr_t H5B2__create_internal(H5B2_hdr_t *hdr, void *parent, H5B2_node_ptr_t *node_ptr, uint16_t depth);

H5_DLL herr_t H5B2__hdr_free(H5B2_hdr_t *hdr);
H5_DLL herr_t H5B2__leaf_free(H5B2_leaf_t *l);
H5_DLL herr_t H5B2__internal_free(H5B2_internal_t *i);

H5_DLL herr_t H5B2__insert(H5B2_hdr_t *hdr, void *udata);
H5_DLL herr_t H5B2__insert_internal(H5B2_hdr_t *hdr, uint16_t depth, unsigned *parent_cache_info_flags_ptr,
                                    H5B2_node_ptr_t *curr_node_ptr, H5B2_nodepos_t curr_pos, void *parent,
                                    void *udata);
H5_DLL herr_t H5B2__insert_leaf(H5B2_hdr_t *hdr, H5B2_node_ptr_t *curr_node_ptr, H5B2_nodepos_t curr_pos,
                                void *parent, void *udata);

H5_DLL herr_t H5B2__update_internal(H5B2_hdr_t *hdr, uint16_t depth, unsigned *parent_cache_info_flags_ptr,
                                    H5B2_node_ptr_t *curr_node_ptr, H5B2_update_status_t *status,
                                    H5B2_nodepos_t curr_pos, void *parent, void *udata, H5B2_modify_t op,
                                    void *op_data);
H5_DLL herr_t H5B2__update_leaf(H5B2_hdr_t *hdr, H5B2_node_ptr_t *curr_node_ptr, H5B2_update_status_t *status,
                                H5B2_nodepos_t curr_pos, void *parent, void *udata, H5B2_modify_t op,
                                void *op_data);

H5_DLL herr_t H5B2__iterate_node(H5B2_hdr_t *hdr, uint16_t depth, H5B2_node_ptr_t *curr_node, void *parent,
                                 H5B2_operator_t op, void *op_data);
H5_DLL herr_t H5B2__node_size(H5B2_hdr_t *hdr, uint16_t depth, H5B2_node_ptr_t *curr_node, void *parent,
                              hsize_t *op_data);

H5_DLL herr_t H5B2__locate_record(const H5B2_class_t *type, unsigned nrec, size_t *rec_off,
                                  const uint8_t *native, const void *udata, unsigned *idx, int *result);
H5_DLL herr_t H5B2__neighbor_internal(H5B2_hdr_t *hdr, uint16_t depth, H5B2_node_ptr_t *curr_node_ptr,
                                      void *neighbor_loc, H5B2_compare_t comp, void *parent, void *udata,
                                      H5B2_found_t op, void *op_data);
H5_DLL herr_t H5B2__neighbor_leaf(H5B2_hdr_t *hdr, H5B2_node_ptr_t *curr_node_ptr, void *neighbor_loc,
                                  H5B2_compare_t comp, void *parent, void *udata, H5B2_found_t op,
                                  void *op_data);

H5_DLL herr_t H5B2__remove_internal(H5B2_hdr_t *hdr, bool *depth_decreased, void *swap_loc, void *swap_parent,
                                    uint16_t depth, H5AC_info_t *parent_cache_info,
                                    unsigned *parent_cache_info_flags_ptr, H5B2_nodepos_t curr_pos,
                                    H5B2_node_ptr_t *curr_node_ptr, void *udata, H5B2_remove_t op,
                                    void *op_data);
H5_DLL herr_t H5B2__remove_leaf(H5B2_hdr_t *hdr, H5B2_node_ptr_t *curr_node_ptr, H5B2_nodepos_t curr_pos,
                                void *parent, void *udata, H5B2_remove_t op, void *op_data);
H5_DLL herr_t H5B2__remove_internal_by_idx(H5B2_hdr_t *hdr, bool *depth_decreased, void *swap_loc,
                                           void *swap_parent, uint16_t depth, H5AC_info_t *parent_cache_info,
                                           unsigned        *parent_cache_info_flags_ptr,
                                           H5B2_node_ptr_t *curr_node_ptr, H5B2_nodepos_t curr_pos, hsize_t n,
                                           H5B2_remove_t op, void *op_data);
H5_DLL herr_t H5B2__remove_leaf_by_idx(H5B2_hdr_t *hdr, H5B2_node_ptr_t *curr_node_ptr,
                                       H5B2_nodepos_t curr_pos, void *parent, unsigned idx, H5B2_remove_t op,
                                       void *op_data);

H5_DLL herr_t H5B2__delete_node(H5B2_hdr_t *hdr, uint16_t depth, H5B2_node_ptr_t *curr_node, void *parent,
                                H5B2_remove_t op, void *op_data);

H5_DLL herr_t H5B2__hdr_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth,
                              const H5B2_class_t *type, haddr_t obj_addr);
H5_DLL herr_t H5B2__int_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth,
                              const H5B2_class_t *type, haddr_t hdr_addr, unsigned nrec, unsigned depth,
                              haddr_t obj_addr);
H5_DLL herr_t H5B2__leaf_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth,
                               const H5B2_class_t *type, haddr_t hdr_addr, unsigned nrec, haddr_t obj_addr);

#ifdef H5B2_DEBUG

H5_DLL herr_t H5B2__assert_internal(hsize_t parent_all_nrec, const H5B2_hdr_t *hdr,
                                    const H5B2_internal_t *internal);
H5_DLL herr_t H5B2__assert_internal2(hsize_t parent_all_nrec, const H5B2_hdr_t *hdr,
                                     const H5B2_internal_t *internal, const H5B2_internal_t *internal2);
H5_DLL herr_t H5B2__assert_leaf(const H5B2_hdr_t *hdr, const H5B2_leaf_t *leaf);
H5_DLL herr_t H5B2__assert_leaf2(const H5B2_hdr_t *hdr, const H5B2_leaf_t *leaf, const H5B2_leaf_t *leaf2);
#endif 

#ifdef H5B2_TESTING
H5_DLL herr_t H5B2__get_root_addr_test(H5B2_t *bt2, haddr_t *root_addr);
H5_DLL int    H5B2__get_node_depth_test(H5B2_t *bt2, void *udata);
H5_DLL herr_t H5B2__get_node_info_test(H5B2_t *bt2, void *udata, H5B2_node_info_test_t *ninfo);
#endif 

#endif 
