
<!-- README.md is generated from README.Rmd. Please edit that file -->

# cardargus <img src="man/figures/logo.svg" align="right" height="139" alt="" />

<!-- badges: start -->

[![CRAN
status](https://www.r-pkg.org/badges/version/cardargus)](https://CRAN.R-project.org/package=cardargus)
[![Lifecycle:
experimental](https://img.shields.io/badge/lifecycle-experimental-orange.svg)](https://lifecycle.r-lib.org/articles/stages.html#experimental)
<!-- badges: end -->

**cardargus** is an R package for creating informative SVG cards with
embedded styles, Google Fonts, badges, and logos. Cards are
self-contained SVG files, ideal for dashboards, reports, and
visualizations.

## Installation

Install from CRAN:

``` r
install.packages("cardargus")
```

Or install the development version from GitHub:

``` r
# install.packages("devtools")
devtools::install_github("StrategicProjects/cardargus")
```

## Basic Example

``` r
library(cardargus)

# Create an informative card
card <- svg_card(
  font = "Jost",
  title = "FAR",
  
  badges_data = list(
    list(label = "Units",         value = "500",     color = "white"),
    list(label = "Federal Grant", value = "$100M",   color = "white"),
    list(label = "State Match",   value = "$80M",    color = "white")
  ),
  
  fields = list(
    list(
      list(label = "Project Name", value = "Boa Vista Residential",
           with_icon = icon_house())
    ),
    list(
      list(label = "Address", value = "123 Flower Street - Downtown")
    ),
    list(
      list(label = "City",   value = "Olinda"),
      list(label = "Region", value = "Pernambuco")
    ),
    list(
      list(label = "Developer",   value = "State Government"),
      list(label = "Contractor",  value = "ABC Construction"), 
      list(label = "Type", value = "PS")  # optional
    )
  ),
  
  bg_color    = "#FF9900",
  title_color = "#fff",
  label_color = "#fff",
  width = 450,
  
  # You can use bundled SVGs or any local file path
  logos = c(
    get_svg_path("seduh.svg"), 
    get_svg_path("morar_bem.svg")
  ),
  logos_height = 40,
  
  bottom_logos = c(
    get_svg_path("tree.svg"), 
    get_svg_path("gov_pe.svg")
  ),
  bottom_logos_height = 40,
  
  footer = paste0(
    "Source: SEDUH/PE on ", 
    format(Sys.time(), "%Y/%m/%d at %H:%M"))
)
```

``` r
include_card_png(card, dpi = 300, width = '50%')
```

<div class="cardargus-card" style="width:50%; max-width:100%;">

<img src="data:image/png;base64,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" alt="Card generated by cardargus" style="width:100%; height:auto;" />

</div>

``` r
# Save as SVG
save_svg(card, "my_card.svg")

# Convert to high-quality PNG
svg_to_png(card, "my_card.png", dpi = 300)
```

## Displaying Cards in R Markdown / Quarto

cardargus provides functions to display cards directly in your
documents:

``` r
# Display card as inline SVG (best quality)
include_card(card)

# Display card as PNG (better compatibility)
include_card_png(card, dpi = 150)

# Save and get path for knitr::include_graphics()
path <- save_card_for_knitr(card, "my_card", format = "png")
knitr::include_graphics(path)
```

For chunk-based workflows, register the cardargus knitr engine:

``` r
# In your setup chunk
register_cardargus_knitr()
```

Then use `cardargus` as a chunk engine:

```` markdown
```{cardargus}`
svg_card(title = "My Card", ...)
```
````

## Features

- 📦 **Self-contained SVG**: All styles and fonts embedded
- 🎨 **Customizable**: Colors, fonts, icons, and layouts
- 🌈 **Gradient Backgrounds**: CSS-style linear gradients support
- 🏷️ **Badges**: Shields.io-style with dynamic colors and uniform height
- 🖼️ **Icons**: Built-in SVG icon library
- 📄 **Export**: High-quality PNG with transparent background
- 🔤 **Google Fonts**: Native support via showtext/sysfonts
- 📊 **R Markdown/Quarto**: Direct display functions

## Gradient Backgrounds

Use CSS-style linear gradients for dynamic card backgrounds:

``` r
# Horizontal gradient
card <- svg_card(
  title = "HOUSING",
  bg_color = "linear-gradient(to right, #1a5a3a, #2e7d32)",
  ...
)

# Diagonal gradient  
card <- svg_card(
  title = "PROGRAM",
  bg_color = "linear-gradient(135deg, #667eea, #764ba2)",
  ...
)

# Vertical gradient
card <- svg_card(
  title = "PROJECT",
  bg_color = "linear-gradient(to bottom, #00c6ff, #0072ff)",
  ...
)
```

Supported directions: `to right`, `to left`, `to top`, `to bottom`, or
angles like `45deg`, `135deg`, `180deg`.

## Custom Cards

``` r
# Define badges
badges <- list(
  list(label = "Units", value = "500", color = "white"),
  list(label = "Status", value = "Active", color = "#4CAF50")
)

# Define fields with custom icon
fields <- list(
  list(
    list(label = "Project", value = "Housing Development")
  ),
  list(
    list(label = "City", value = "Pesqueira"),
    list(label = "State", value = "Pernambuco")
  )
)

# Create card with logos
card <- svg_card(
  title = "HOUSING",
  badges_data = badges,
  fields = fields,
  bg_color = "#2c3e50",
  title_color = "#ecf0f1",
  width = 200,
  logos = c(get_svg_path("morar_bem.svg")),
  logos_height = 40,
)
```

<div class="cardargus-card" style="width:25%; max-width:100%;">

<img src="data:image/png;base64,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" alt="Card generated by cardargus" style="width:100%; height:auto;" />

</div>

## Bundled SVGs

``` r
# List available SVGs
list_bundled_svgs()

# Get full path
get_svg_path("morar_bem.svg")
```

## Available Icons

``` r
# Built-in icons
icon_house()        # House (default)
icon_building()     # Building
icon_construction() # Construction
icon_map_pin()      # Location
icon_money()        # Money

# Or use your own SVG file
# with_icon = "/path/to/custom_icon.svg"
```

## Font Setup

For best font rendering:

``` r
# Setup Google Fonts (recommended)
setup_fonts()

# Or download fonts for offline use
install_fonts()

# Check font availability
font_available("Jost")
```

## Chrome Rendering (Recommended)

For **perfect font rendering** with Google Fonts, use headless Chrome:

``` r
# Check if Chrome is available
chrome_available()

# If Chrome is not installed, download it automatically (~150MB)
ensure_chrome(download = TRUE)

# Convert to PNG with Chrome (best quality)
svg_to_png_chrome(card, "my_card.png", dpi = 300)

# Convert to PDF (vector output)
svg_to_pdf_chrome(card, "my_card.pdf")

# In R Markdown / Quarto - force Chrome engine
include_card_png(card, dpi = 300, engine = "chrome")
```

Install `chromote` for Chrome support:

``` r
install.packages("chromote")
```

## Authors

- **Andre Leite** - *Maintainer* - <leite@castlab.org>
- **Hugo Vasconcelos** - <hugo.vasconcelos@ufpe.br>
- **Diogo Bezerra** - <diogo.bezerra@ufpe.br>

## License

MIT License - see [LICENSE](LICENSE) for details.
